import { Injectable } from '@angular/core';
import { APIService } from '@app/shared/services/api.service';
import { AnnounceTXParams } from '@app/shared/services/wallet/types';
import { sc, tx, wallet } from '@cityofzion/neon-core';
import Decimal from 'decimal.js';

@Injectable({ providedIn: 'root' })
export class TransactionService {
  constructor(private readonly apiService: APIService) {}

  public async createTransfer(
    fromAddress: string,
    toAddress: string,
    assetHash: string,
    amount: number | string | Decimal
  ): Promise<AnnounceTXParams> {
    const dvgDecimals = 8; // TODO fetch from backend

    const script = sc.createScript({
      scriptHash: assetHash,
      operation: 'transfer',
      args: [
        sc.ContractParam.hash160(fromAddress),
        sc.ContractParam.hash160(toAddress),
        amount.toString() as unknown as sc.ContractParam, // official example uses number type but TypeScript shows error
        sc.ContractParam.any(),
      ],
    });

    const validUntilBlock = (await this.apiService.blockCount()) + 1000;
    const signers = [
      {
        account: wallet.getScriptHashFromAddress(fromAddress),
        scopes: tx.WitnessScope.CalledByEntry,
      },
    ];
    const serializedTransactionLength = new tx.Transaction({
      signers,
      validUntilBlock,
      script,
    }).serialize().length;

    const fees = await this.apiService.fees(script, serializedTransactionLength);

    return {
      fromAddress,
      toAddress,
      asset: assetHash,
      amount: String(amount),
      fee: new Decimal(fees.systemFee).plus(fees.networkFee).div(new Decimal(10).pow(dvgDecimals)).toString(),
      txInstance: new tx.Transaction({
        signers,
        script,
        validUntilBlock,
        networkFee: fees.networkFee,
        systemFee: fees.systemFee,
      }),
    };
  }
}
