import Decimal from 'decimal.js';
import { AbstractControl, ValidatorFn } from '@angular/forms';
import { CountryCode, parsePhoneNumber } from 'libphonenumber-js';
import * as countries from 'i18n-iso-countries';

export class CustomValidators {
    public static match(equalTo: AbstractControl): ValidatorFn {
        return (control: AbstractControl): { [key: string]: any } => {
            const val = control.value;

            if (val === undefined || val === null || val.trim() === '') {
                // won't check if empty
                return null;
            }

            if (val !== equalTo.value) {
                return { match: true };
            }

            return null;
        };
    }

    public static isValidCountry(codeKey: string): ValidatorFn {
        return (control: AbstractControl): { [key: string]: any } => {
            const val = control.value;

            if (val === undefined || val === null) {
                // won't check if empty
                return null;
            }

            if (typeof val === 'object' && !countries.getName(val[codeKey], 'en')) {
                return { country: true };
            } else if (typeof val === 'string') {
                return { country: true };
            }

            return null;
        };
    }

    public static isValidPhoneNumber(countryInput: AbstractControl): ValidatorFn{
        return (control: AbstractControl): { [key: string]: any } => {
            const val = control.value;

            if (val === undefined || val === null || val.trim() === '') {
                // won't check if empty
                return null;
            }

            const country = countryInput.value;
            if (country === undefined || country === null || country.trim() === '') {
                return null;
            }

            const contryCode =  countries.getAlpha2Code(country, 'en') as CountryCode;
            if (contryCode === undefined || contryCode === null || contryCode.trim() === '') {
                return { invalidPhoneNumber: true };
            }

            let phoneNumber = null;
            try {
                phoneNumber = parsePhoneNumber(val, contryCode);
            } catch (err) {}

            if (!phoneNumber || !phoneNumber.isValid() || val !== phoneNumber.nationalNumber) {
                return { invalidPhoneNumber: true };
            }

            return null;
        };
    }

    public static enoughFunds(balance: number): ValidatorFn {
        return (control: AbstractControl) => {
            const value = control.value;

            if (value === undefined || value === null) {
                return null;
            }

            if (new Decimal(balance).lessThan(value)) {
                return { notEnoughFunds: true };
            }

            return null;
        };
    }
}
