import { KycForm } from './../types/KycForm';
import { Turnover } from './../types/Turnover';
import { Injectable } from '@angular/core';
import { Subscription, Observable } from 'rxjs';
import { SocketSubServiceResolvable } from '@app/core/socket/SocketSubServiceResolvable';
import { User } from '../types/User';
import { SocketService } from '@app/core/socket/SocketService';
import { isEmpty } from '../Helper';
import { HttpClient } from '@angular/common/http';
import { BankAccount } from '../types/BankAccount';
import { KycTier } from '../objects/KycTier';
import { Product } from '../types/Product';

@Injectable()
export class UserService extends SocketSubServiceResolvable<Partial<User>>{
    public readonly user: User = { email: null };
    public showBalance = false;
    public maintenanceMode: {mode: boolean, reason: string} = {mode: false, reason: null};
    public kycForm: KycForm = null;

    constructor(
        private readonly httpClient: HttpClient,
        protected readonly socketService: SocketService
    ) {
        super('user', socketService);
    }

    // Getters
    public get(key: (keyof User)): any {
        if (!isEmpty(this.user[key])) {
            return this.user[key];
        }

        return this.user[key] || null;
    }

    public resolve(): Promise<Partial<User>> {
        return super.resolve();
    }

    public async reload(): Promise<Partial<User>> {
        super.onClear();
        return this.resolve();
    }

    private startLoadData(): void {
        this.socket.once('dataR', (data: Partial<User>) => {
            if (!data) {
                this.eventSubject.next({ name: 'data_loaded', args: [this.user] });
                return;
            }

            this.onClear();
            this.onUpdate(data);

            this.eventSubject.next({ name: 'data_loaded', args: [this.user] });
        });
        this.socket.emit('data');
    }

    protected onConnect(): void {
        this.startLoadData();
    }

    protected onClear(): void {
        super.onClear();
        for (const key of Object.keys(this.user)) {
            if (key === 'email') {
                this.user[key] = null;
            } else {
                delete this.user[key];
            }
        }
    }

    private onUpdate(data: Partial<User>): void {
        for (const key of Object.keys(data)) {

            if (isEmpty(data[key])) {
                continue;
            }
            if (key === 'birthDate') {
                this.user.profile.birthDate = new Date(data[key]);
            } else {
                this.user[key] = data[key];
            }
        }

        // lets set default avatar here
        if (isEmpty(this.user.profile.avatar)) {
            this.user.profile.avatar = '/assets/images/user-blank.jpg';
        }

        this.eventSubject.next({ name: 'update' });
    }

    private onUpdateBank(data: Partial<BankAccount>): void {
        if (!this.user.bankAccount) {
            this.user.bankAccount = {};
        }

        for (const key of Object.keys(data)) {
            if (isEmpty(data[key])) {
                continue;
            }
            if (key === 'createdDate' || key === 'updatedDate') {
                this.user.bankAccount[key] = new Date(data[key]);
            } else {
                this.user.bankAccount[key] = data[key];
            }
        }

        this.eventSubject.next({ name: 'update-bank' });
    }

    private onDeleteBank(key: number, data: Partial<BankAccount>): void {
        if (this.user.bankAccount && this.user.bankAccount.id === key) {
            this.user.bankAccount = null;
            this.eventSubject.next({ name: 'update-bank' });
        }
    }

    public onUpdated(cb: () => void): Subscription {
        return this.onEvent('update', cb);
    }

    public onBankUpdated(cb: () => void): Subscription {
        return this.onEvent('update-bank', cb);
    }

    private onKycUpdate(data: Partial<KycTier>): void {
        if (!this.user.kyc) {
            this.user.kyc = [];
        }

        let existing = this.kycTier(data.tier);
        if (!existing) {
            existing = new KycTier();
            this.user.kyc.push(existing);
        }

        for (const key of Object.keys(data)) {
            if (isEmpty(data[key])) {
                continue;
            }
            if (key === 'createdDate' || key === 'updatedDate') {
                existing[key] = new Date(data[key]);
            } else if (key === 'final') {
                existing[key] = Boolean(data[key]);
            } else {
                existing[key] = data[key];
            }
        }

        this.eventSubject.next({ name: 'update-kyc' });
    }

    private onKycInsert(kycTier: KycTier): void {
        if (!this.user.kyc) {
            this.user.kyc = [];
        }

        let existing = this.kycTier(kycTier.tier);
        if (!existing) {
            existing = new KycTier();
            existing.tier = kycTier.tier;
            existing.status = kycTier.status;
            existing.createdDate = kycTier.createdDate;
            existing.updatedDate = kycTier.updatedDate;
            existing.rejectionReason = kycTier.rejectionReason;
            existing.final = kycTier.final;

            this.user.kyc.push(existing);
        }

        this.eventSubject.next({ name: 'insert-kyc' });
    }

    // private listeners
    protected initListeners(): void {
        this.socket.on('update', this.onUpdate.bind(this));
        this.socket.on('update-bank', this.onUpdateBank.bind(this));
        this.socket.on('delete-bank', this.onDeleteBank.bind(this));
        this.socket.on('update-kyc', this.onKycUpdate.bind(this));
        this.socket.on('insert-kyc', this.onKycInsert.bind(this));
        this.socket.on('delete-kyc', this.onKycFinalReject.bind(this));
        this.socket.on('maintenance', this.setMaintenanceMode.bind(this));
    }

    public getKycTier(): any {
        return this.user ? this.user.kycTier : null;
    }

    public kycTier(tier: number): KycTier {
        const kyc = null;

        if (this.user && this.user.kyc && this.user.kyc.length > 0) {
            return this.user.kyc.find(r => r.tier === tier);
        }

        return kyc;
    }

    public getKycStatus(): any {
        return this.user ? this.user.kycVerificationStatus : null;
    }

    public setShowBalance(showBalance: boolean): void {
        this.showBalance = showBalance;
        // localStorage.setItem('showBalance', this.showBalance ? 'true' : 'false');
    }

    public getDefaultShowBalance(): boolean {
        return this.user ? Boolean(this.user.hideValues) : false;
    }

    public getWithdrawWhitelistEnabled(): boolean {
        return Boolean(this.user.whitelistEnabled);
    }

    public setDefaultShowBalance(): void {
        if (Boolean(this.user.hideValues) === true) {
            this.showBalance = false;
        } else {
            this.showBalance = true;
        }

        // if (localStorage.getItem('showBalance')) {
        //     localStorage.removeItem('showBalance');
        // }
    }

    setMaintenanceMode(data: {mode: boolean, reason: string}): void {
        if (!data.mode && data.mode !== false) {
            return;
        }

        this.maintenanceMode = data;
    }

    public getMe(): Observable<any> {
        return this.httpClient.get<Turnover[]>('/user/me');
    }

    public getUserTurnover(): Observable<any> {
        return this.httpClient.get<Turnover[]>('/users/me/turnover');
    }

    public showKycTierForm(tierNr: number, prevTier: number, kycPayment: Product): boolean {
        if (!kycPayment) {
            return false;
        }

        const kycTier: KycTier = this.kycTier(tierNr);

        if (kycTier == null && this.user.kycTier !== prevTier) {
            return true;
        } else if (kycTier == null && kycPayment.disabled === 1) {
            return true;
        } else if (kycTier != null && kycTier.status === 'PAID') {
            return true;
        }

        return false;
    }

    private onKycFinalReject(data: Partial<KycTier>): void {
        const index = this.user.kyc.findIndex(k => k.tier === data.tier);

        if (index > -1){
            this.user.kyc.splice(index, 1);
        }
    }
}
