import { Injectable, EventEmitter } from '@angular/core';
import { SocketSubServiceResolvable } from '@app/core/socket/SocketSubServiceResolvable';
import Transaction from '../objects/Transaction';
import { SocketService } from '@app/core/socket/SocketService';
import { AccountService } from './AccountService';
import { HttpClient } from '@angular/common/http';
import { isEmpty } from '../Helper';
import Account from '../objects/Account';
import { Observable } from 'rxjs';
import { SwalService } from './SwalService';
import { Decimal as DecimalJS } from 'decimal.js';

@Injectable()
export class TransactionService extends SocketSubServiceResolvable<Transaction[]>{
  protected socket: SocketIOClient.Socket;
  public transactions: Transaction[] = [];
  public transactionUpdated: EventEmitter<Transaction> = new EventEmitter<Transaction>();

  constructor(
    private readonly httpClient: HttpClient,
    protected readonly socketService: SocketService,
    public readonly accountService: AccountService,
    private readonly swalService: SwalService
  ) {
    super('transactions', socketService);
    this.initOutListeners();
  }

  public resolve(): Promise<Transaction[]> {
    return super.resolve();
  }

  private startLoadData(): void {
    this.socket.once('dataR', (data: any[]) => {
      this.transactions.length = 0;

      if (isEmpty(data)){
        this.eventSubject.next({name: 'data_loaded', args: [this.transactions]});
        return;
      }

      for (const transaction of data){
        this.addOrUpdate(transaction, true);
      }

      this.eventSubject.next({name: 'data_loaded', args: [this.transactions]});
    });
    this.socket.emit('data');
  }

  private addOrUpdate(transaction: any, descOrder: boolean = false): Transaction {
    let existing = this.transactions.find(t => t.id === transaction.id);

    if (!existing) {
      existing = new Transaction(this);

      if (descOrder) {
        this.transactions.push(existing);
      } else {
        this.transactions.unshift(existing);
      }
    }

    existing.id = transaction.id;
    existing.number = transaction.number;
    existing.addressTo = transaction.addressTo;
    existing.addressFrom = transaction.addressFrom;
    existing.transactionId = transaction.transactionId;
    existing.accountFromId = transaction.accountFromId;
    existing.accountToId = transaction.accountToId;
    existing.type = transaction.type;
    existing.subType = transaction.subType;
    existing.status = transaction.status;
    existing.amountTo = transaction.amountTo;
    existing.amountToEur = transaction.amountToEur;
    existing.amountToUsd = transaction.amountToUsd;
    existing.amountFrom = transaction.amountFrom;
    existing.amountFromEur = transaction.amountFromEur;
    existing.amountFromUsd = transaction.amountFromUsd;
    existing.systemFee = transaction.systemFee;
    existing.txFee = transaction.txFee;
    existing.total = transaction.total;
    existing.totalEur = transaction.totalEur;
    existing.totalUsd = transaction.totalUsd;
    existing.currencyCodeFrom = transaction.currencyCodeFrom;
    existing.currencyCodeTo = transaction.currencyCodeTo;
    existing.rate = transaction.rate;
    existing.rateEur = transaction.rateEur || 0;
    existing.rateUsd = transaction.rateUsd || 0;
    existing.createdDate = new Date(transaction.createdDate);
    existing.updatedDate = new Date(transaction.updatedDate);
    existing.note = transaction.note;
    existing.beneficiary = transaction.beneficiary;
    existing.iban = transaction.iban;

    return existing;
  }

  protected onConnect(): void {
    this.startLoadData();
  }

  protected onClear(): void {
    super.onClear();
    this.transactions.length = 0;
  }

  // public getters
  public get all(): Transaction[] {
    return this.transactions;
  }

  public transaction(id: number): Transaction{
    return this.transactions.find(a => a.id === id);
  }

  public getByAccount(id: number | Account): Transaction[]{
    if (typeof id === 'number') {
      return this.transactions.filter(a => a.accountFromId === id || a.accountToId === id);
    }

    if (!id.id) {
      return null;
    }

    return this.transactions.filter(a => a.accountFromId === id.id || a.accountToId === id.id);
  }

  // private listeners
  protected initListeners(): void {
    this.socket.on('update', (transactionId: number, changed: Partial<any>) => {

      let transaction = this.transaction(transactionId);
      if (!transaction) {
        transaction = this.addOrUpdate(changed);
        this.transactionUpdated.emit(transaction);
        this.removeExtraTransactions(transaction);
        return;
      }

      for (const key of Object.keys(changed)) {
        transaction[key] = changed[key];
      }

      this.showToast(transaction);
      this.transactionUpdated.emit(transaction);
    });

    this.socket.on('new', (transactionId: number, transaction: Transaction) => {
      if (isEmpty(transaction) || isEmpty(transaction.id)) {
        return;
      }

      this.addOrUpdate(transaction);
      this.showToast(transaction);
      this.transactionUpdated.emit(transaction);
      this.removeExtraTransactions(transaction);
    });
  }

  private showToast(transaction: Transaction): void  {
    if (transaction.confirmed === false) {
      return;
    }

    const titleParams = {
      refNumber: transaction.number,
      code: transaction.currencyCodeFrom,
      codeTo: transaction.currencyCodeTo,
      amount: new DecimalJS(transaction.total).toNumber()
    };

    if (transaction.type === 'CRYPTO') {
      if (transaction.subType === 'RECEIVE') {
        if (transaction.status === 3) {
          this.swalService.showToast({title: 'SWAL.TOAST_RECEIVE_TX_PENDING', titleParams, type: 'info', timer: 10000});
        } else if (transaction.status === 5) {
          this.swalService.showToast({title: 'SWAL.TOAST_RECEIVE_TX_COMPLETE', titleParams, type: 'success', timer: 10000});
        }
      } else if (transaction.subType === 'EXCHANGE') {
        if (transaction.status === 5) {
          this.swalService.showToast({title: 'SWAL.TOAST_EXCHANGE_TX_COMPLETE', titleParams, type: 'success', timer: 10000});
        }
      } else if (transaction.subType === 'WITHDRAW') {
        if (transaction.status === 5) {
          this.swalService.showToast({title: 'SWAL.TOAST_WITHDRAW_TX_COMPLETE', titleParams, type: 'success', timer: 10000});
        }
      }
    }
  }

  private removeExtraTransactions(transaction: Transaction): void {
    const MAX_TRANSACTIONS = 10;

    const byAccount = this.transactions.filter(
      t => t.accountFromId === transaction.accountFromId ||
      t.accountToId === transaction.accountToId);

    if (byAccount.length <= MAX_TRANSACTIONS) {
      return;
    }

    const ind = this.transactions.findIndex(t => t.id === byAccount[byAccount.length - 1].id);
    if (!ind || ind === -1) {
      return;
    }

    this.transactions.splice(ind, 1);
  }

  // other listeners
  private initOutListeners(): void {
  }

  /** API ROUTES */
  public get latest(): Observable<any[]> {
    return this.httpClient.get<any>('/transactions/latest');
  }

  public list(from: number, take: number, accountId?: number): Observable<any[]> {
    const getOptions =  {
      params: {
        from: String(from),
        take: String(take),
        accountId: accountId ? String(accountId) : null
      }
    };
    return this.httpClient.get<any>('/transactions/list', getOptions);
  }
}
