import { Injectable } from '@angular/core';
import Swal, { SweetAlertResult } from 'sweetalert2';
import { HttpErrorResponse } from '@angular/common/http';
import { TranslateService } from '@ngx-translate/core';
import { ErrorTranslate } from '../types/ErrorTranslate';

@Injectable()
export class SwalService {
    constructor(
        private readonly translateService: TranslateService
    ) {
    }

    public showSuccess({
        title = 'SWAL.SUCCESS',
        titleParams = null,
        text = null,
        textParams = null,
        timer = 3000,
        toast = false,
        html = null,
        htmlParams = null
    }: {
        title?: string,
        titleParams?: any,
        text?: string,
        textParams?: any,
        timer?: number,
        toast?: boolean,
        html?: string,
        htmlParams?: any
    } = {}): Promise<SweetAlertResult> {
        return Swal.fire({
            position: toast ? 'top' : 'center',
            title,
            text,
            html,
            timer,
            showConfirmButton: timer == null,
            toast,
            heightAuto: false,
        });
    }

    public showInfo({
        title = 'SWAL.SUCCESS',
        titleParams = null,
        text = null,
        textParams = null,
        timer = 3000,
        toast = false
    }: {
        title?: string,
        titleParams?: any,
        text?: string,
        textParams?: any,
        timer?: number,
        toast?: boolean
    } = {}): Promise<SweetAlertResult> {
        return Swal.fire({
            position: toast ? 'top' : 'center',
            title: title ? title : this.translateService.instant(title, titleParams),
            text: text ? this.translateService.instant(text, textParams) : null,
            timer,
            showConfirmButton: timer == null,
            toast,
            heightAuto: false,
        });
    }

    public showToast({
        title = 'SWAL.SUCCESS',
        titleParams = null,
        type = 'success',
        timer = 2000
    }: {
        title?: string,
        titleParams?: any,
        type?: 'success' | 'error' | 'info',
        timer?: number
    } = {}): Promise<SweetAlertResult> {
        return Swal.fire({
            position: 'top',
            title: this.translateService.instant(title, titleParams),
            timer,
            toast: true,
            showConfirmButton: false,
        });
    }

    public showError({
        title = 'SWAL.SUCCESS',
        titleParams = null,
        text = null,
        textParams = null,
        html = null,
        htmlParams = null,
        timer = null
    }: {
        title?: string,
        titleParams?: any,
        text?: string,
        textParams?: any,
        html?: string,
        htmlParams?: any,
        timer?: number
    } = {}): Promise<SweetAlertResult> {
        return Swal.fire({
            position: 'center',
            title,
            text,
            html,
            timer,
            heightAuto: false,
        });
    }

    public showConfirm({
        title = 'SWAL.ARE_YOU_SURE',
        titleParams = null,
        text = null,
        textParams = null,
        html = null,
        htmlParams = null
    }: {
        title?: string,
        titleParams?: any,
        text?: string,
        textParams?: any,
        html?: string,
        htmlParams?: any
    } = {}): Promise<SweetAlertResult> {
        return Swal.fire({
            position: 'center',
            text: text ? this.translateService.instant(text, textParams) : null,
            html: html ? this.translateService.instant(html, htmlParams) : null,
            title: this.translateService.instant(title, titleParams),
            showCancelButton: true,
            confirmButtonText: this.translateService.instant('BUTTON.CONFIRM'),
            cancelButtonText: this.translateService.instant('BUTTON.CANCEL'),
            heightAuto: false,
        });
    }

    public showException(err: any, customErrorParams?: ErrorTranslate | ErrorTranslate[]): Promise<SweetAlertResult> {
        if (err instanceof HttpErrorResponse && err.status === 417) {
            const errorCode: string = err.error ? err.error.error : null;
            const errorParam: string = err.error ? err.error.error_param : null;
            const errorValue: string = err.error ? err.error.param_value : null;
            let errorMessage: string = null;

            if (errorParam === 'CUSTOM_MESSAGE'){
                return Swal.fire({
                    position: 'center',
                    title: this.translateService.instant('SWAL.' + errorCode.toUpperCase()),
                    text: errorValue,
                    heightAuto: false,
                });
            }

            if (customErrorParams) {
                let customErrors: ErrorTranslate[];
                let params: any = {};

                if (!Array.isArray(customErrorParams)) {
                    customErrors = [customErrorParams];
                } else {
                    customErrors = customErrorParams;
                }

                for (const customError of customErrors) {
                    if (errorCode !== customError.key) {
                        continue;
                    }
                    if (customError.keyParam && errorParam !== customError.keyParam) {
                        continue;
                    }

                    customError.errorParams = customError.errorParams || {};
                    params = customError.params || {};

                    for (const errorParamKey of Object.keys(customError.errorParams)) {
                        if (errorParam === errorParamKey) {
                            params.param = this.translateService.instant(customError.errorParams[errorParamKey]);
                        }
                    }

                    if (errorValue) {
                        params.param_value = errorValue;
                    }

                    return Swal.fire({
                        position: 'center',
                        title: this.translateService.instant(customError.title || 'ERROR.SERVER_ERROR'),
                        text: this.translateService.instant(customError.message, params),
                        heightAuto: false,
                    });
                }
            }

            if (errorCode === 'error' && errorParam) {
                errorMessage = this.translateService.instant('ERROR.' + errorParam.toUpperCase());
            } else if (errorCode && errorParam) {
                errorMessage = this.translateService.instant('ERROR.FIELD_ERROR', { field: errorParam, message: errorCode });
            } else {
                errorMessage = this.translateService.instant('ERROR.SOMETHING_WENT_WRONG', { message: err.message });
            }

            return Swal.fire({
                position: 'center',
                title: this.translateService.instant('ERROR.SERVER_ERROR'),
                text: errorMessage,
                heightAuto: false,
            });
        }

        return Swal.fire({
            position: 'center',
            title: this.translateService.instant('ERROR.SERVER_ERROR'),
            text: this.translateService.instant('ERROR.SOMETHING_WENT_WRONG', { message: err.statusText }),
            heightAuto: false,
        });
    }
}
