import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { Logger } from '@app/core/LoggerService';
import { Decimal } from 'decimal.js';
import { Observable, of, throwError } from 'rxjs';
import { map } from 'rxjs/operators';

export interface CryptoMarketRate {
  pair: string;
  ticker: string;
  open: number;
  average: number;
  last: number;
  reversed: boolean;
  timestamp: number;
  datetime: Date;
}

export interface CryptoMarketOhlcEntry {
  time: number;
  open: string;
  high: string;
  low: string;
  close: string;
  vwap: string;
  volume: string;
  count: number;
}

export interface CryptoMarketOhlc {
  pair: string;
  interval: string;
  since: number;
  entries: CryptoMarketOhlcEntry[];
  timestamp: number;
}

export enum OhlcInterval {
  oneMinute = 'oneMinute',            // 1
  fiveMinutes = 'fiveMinutes',        // 5
  fifteenMinutes = 'fifteenMinutes',  // 15
  thirtyMinutes = 'thirtyMinutes',    // 30
  oneHour = 'oneHour',                // 60
  fourHours = 'fourHours',            // 240
  oneDay = 'oneDay',                  // 1440
  sevenDays = 'sevenDays',            // 10080
  fifteenDays = 'fifteenDays'         // 21600
}

//
// ExchangeRatesService
//
// TODO: Create Refresh and Live update via WebSockets
//
@Injectable()
export class ExchangeRatesService {
    private readonly logger = new Logger(ExchangeRatesService.name);
    private exchangeRatesMap = new Map<string, CryptoMarketRate>();

    constructor(
      private readonly httpClient: HttpClient) {
        // Async, so safe to call in ctor
        this.logger.info(`Warming-up All currency rates...`);
        this.updateAllRatesObservable().subscribe(rates => {
            this.logger.info(`Warming-up All currency rates: Received.`);
          },
          err => {
            this.logger.info(`Warming-up Error: ${err.message || err}`);
          },
          () => {
            this.logger.info(`Warming-up All currency rates: Done.`);
          }
        );
    }

    //
    // EXCHANGE RATE - Async version: returns Promise to a new value or error
    //
    public async getExchangeValueAsync(fromCode: string, toCode: string, value: string | number): Promise<number> {
      return this.getExchangeValueObservable(fromCode, toCode, value).toPromise();
    }

    //
    // EXCHANGE RATE - Observable version: returns Observable to a new value or error
    //
    public getExchangeValueObservable(fromCode: string, toCode: string, value: string | number): Observable<number> {
      const fixedLenght = 8;
      return this.getExchangeRateObservable(fromCode, toCode).pipe(
        map(rate => new Decimal(value).mul(rate.average).toDecimalPlaces(fixedLenght).toNumber())
      );
    }

    //
    // Async version: returns Promise to a new value or error
    //
    public getExchangeRateAsync(fromCode: string, toCode: string): Promise<CryptoMarketRate> {
      return this.getExchangeRateObservable(fromCode, toCode).toPromise();
    }

    //
    // Observable version: returns Observable to a new value or error
    //
    public getExchangeRateObservable(fromCode: string, toCode: string): Observable<CryptoMarketRate> {
      if (!this.exchangeRatesMap || Object.keys(this.exchangeRatesMap).length <= 0) {
        // Start updating cycle
        // tslint:disable-next-line: deprecation
        return new Observable(observer => {
          this.updateAllRatesObservable().subscribe(
            exchangeRatesMap => {
              const rate = exchangeRatesMap[`${fromCode}${toCode}`];
              if (rate) {
                observer.next(rate);
              }
              else {
                this.logger.warn(`Rates not found, returning undefined for ${fromCode} -> ${toCode}`);
                const error = new Error(`Rate not found: ${fromCode} -> ${toCode}`);
                observer.error(error);
              }
            },
            err => {
              this.logger.error(`Error: ${err}`, err.stack);
              observer.error(err);
            }
          );
        });
      }
      else {
        const rate = this.exchangeRatesMap[`${fromCode}${toCode}`];
        if (rate) {
          return of(rate);
        }
        else {
          this.logger.warn(`Rates not found, returning undefined for ${fromCode} -> ${toCode}`);
          return throwError(new Error(`Rate not found: ${fromCode} -> ${toCode}`));
        }
      }
    }

    //
    // OHLC - Observable version: returns Observable to a new value or error
    //
    public getOhlcObservable(base: string, quote: string, interval?: string, since?: number): Observable<CryptoMarketOhlc> {
      return this.httpClient.get<CryptoMarketOhlc>(`/crypto-market-rates/ohlc/${base}/${quote}/${interval ?? ''}/${since ?? ''}`);
    }

    //
    // Privates
    //

    //
    // Receive from API
    //
    private updateAllRatesObservable(): Observable<Map<string, CryptoMarketRate>> {
      return new Observable(observer => {
        this.getAllRatesObservable().subscribe(
          pairs => {
            // Enrich DTO
            const normalizedPairs = pairs.map(p => {
              p.datetime = new Date(p.timestamp);
              return p;
            });
            // Convert to map
            this.exchangeRatesMap = normalizedPairs.reduce((acc, current) => {
                acc[current.pair] = current;
                return acc;
              },
              new Map<string, CryptoMarketRate>()
            );
            this.logger.debug(`Got pairs: ${JSON.stringify(this.exchangeRatesMap)}`);
            observer.next(this.exchangeRatesMap);
          },
          err => {
            this.logger.error(`Error: ${err}`, err.stack);
            observer.error(err);
          }
        );
      });
    }

    private getAllRatesObservable(): Observable<CryptoMarketRate[]> {
      return this.httpClient.get<CryptoMarketRate[]>('/crypto-market-rates/pairs');
    }
}
