import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { SocketService } from '@app/core/socket/SocketService';
import { SocketSubServiceResolvable } from '@app/core/socket/SocketSubServiceResolvable';
import { Observable, Subscription } from 'rxjs';
import { isEmpty } from '../Helper';
import { Currency } from '../objects/Currency';
import { CurrencyRates } from '../types/CurrencyRates';

@Injectable()
export class CurrencyService extends SocketSubServiceResolvable<Currency[]> {
    protected socket: SocketIOClient.Socket;
    private defaultCurrency = 'BTC';
    readonly currencies: Currency[] = [];

    constructor(
      protected readonly socketService: SocketService,
      private readonly httpClient: HttpClient
    ) {
      super('currency', socketService);
    }

    //
    // Public getters
    //

    public get all(): Currency[]{
      return this.currencies;
    }

    //
    // Public methods
    //
    public resolve(): Promise<Currency[]>{
      return super.resolve();
    }

    public currency(code: string = this.defaultCurrency): Currency {
      return this.currencies.find(c => c.code === code.toUpperCase());
    }

    public getByType(type: string): Currency[] {
      return this.currencies.filter(c => c.type === type.toLowerCase());
    }

    public onRateChange(cb: (code?: string) => void): Subscription;
    public onRateChange(cb: (code?: string) => void, once: boolean): void;
    public onRateChange(cb: (code?: string) => void, once?: boolean): Subscription | void{
      return this.onEvent('rates', cb, once);
    }

    // API calls
    public getHistory(code: string, show: 'year' | '3month' | 'month' | 'week' | 'day' | 'hour'): Observable<any> {
      return this.httpClient.get<any[]>('/currency/history/' + code + '/' + show);
    }

    //
    // Protected listeners
    //

    protected onConnect(): void {
      this.startLoadData();
    }

    protected onClear(): void {
      super.onClear();
      this.currencies.length = 0;
    }

    protected initListeners(): void {
      this.socket.on('rate', this.onRate.bind(this));
    }

    private startLoadData(): void {
      this.socket.once('dataR', (data: {currencies: any[], default: string}) => {
        if (!data){
          this.eventSubject.next({name: 'data_loaded', args: [this.currencies]});
          return;
        }

        if (data.default){
          this.defaultCurrency = data.default;
        }

        for (const curr of data.currencies){
          let existing = this.currency(curr.code);
          if (!existing){
            existing = new Currency(this);
            this.currencies.push(existing);
          }
          existing.code = curr.code;
          existing.codeLabel = curr.codeLabel;
          existing.name = curr.name;
          existing.protocol = curr.protocol;
          existing.symbol = curr.symbol;
          existing.type = curr.type;
          this.onRate(curr.code, curr.rates);
        }

        // update from for each received again as now we should have all initial rates
        for (const curr of data.currencies){
          this.updateFrom(curr.code);
        }

        this.eventSubject.next({name: 'data_loaded', args: [this.currencies]});
      });
      this.socket.emit('data');
    }

    private onRate(curr: string | Currency, rates: CurrencyRates): void {
      if (typeof curr === 'string'){
        curr = this.currency(curr);
      }

      if (!curr || isEmpty(rates)){
        return;
      }

      // update rates
      for (const c of Object.keys(rates)){
        curr.rates[c] = rates[c];
        if (curr.to.indexOf(c) === -1){
          if (this.currency(c)){
            curr.to.push(c);
          }
        }
      }

      this.updateFrom(curr);
      this.eventSubject.next({name: 'rates', args: [curr.code]});
    }

    private updateFrom(curr: string | Currency): void {
      if (typeof curr === 'string'){
        curr = this.currency(curr);
      }

      for (const c of this.currencies){
        if (c.code === curr.code || curr.from.indexOf(c.code) !== -1){
          continue;
        }
        if (c.to.indexOf(curr.code) !== -1){
          curr.from.push(c.code);
        }
      }
    }
}
