import { HttpClient } from '@angular/common/http';
import { EventEmitter, Injectable } from '@angular/core';
import { SocketService } from '@app/core/socket/SocketService';
import { SocketSubServiceResolvable } from '@app/core/socket/SocketSubServiceResolvable';
import { Decimal as DecimalJS } from 'decimal.js';
import { Observable } from 'rxjs';
import { isEmpty } from '../Helper';
import Account from '../objects/Account';
import { CurrencyService } from './CurrencyService';

@Injectable()
export class AccountService extends SocketSubServiceResolvable<Account[]> {
    protected socket: SocketIOClient.Socket;
    readonly accounts: Account[] = [];

    public newAccountAdded: EventEmitter<Account> = new EventEmitter<Account>();
    public totals: { EUR: number, USD: number } = { EUR: 0, USD: 0 };

    constructor(
        private readonly httpClient: HttpClient,
        public readonly currencyService: CurrencyService,
        protected readonly socketService: SocketService,
    ) {
        super('account', socketService);
        this.initOutListeners();
    }

    public resolve(): Promise<Account[]> {
        return super.resolve();
    }

    public get all(): Account[] {
        return this.accounts;
    }

    public account(id: number): Account {
        return this.accounts.find(a => a.id === id);
    }

    public getByCurrency(code: string): Account[] {
        return this.accounts.filter(a => a.currencyCode === code);
    }

    public getByType(code: string): Account[] {
        return this.accounts.filter(a => a.type === code);
    }

    public getByCurrencies(codes: string[]): Account[] {
        return this.accounts.filter(a => codes.indexOf(a.currencyCode) !== -1);
    }

    public getDefaultAccount(): Account {
        return this.accounts.find(a => a.currencyCode === 'BTC');
    }
    public getAccountByCurrency(code: string): Account {
        return this.accounts.find(a => a.currencyCode === code);
    }

    public createAccount(currencyCode: string, label: string): Observable<any> {
        return this.httpClient.post('/accounts', { currencyCode, label });
    }

    // private listeners
    protected initListeners(): void {
        this.socket.on('update', (accountId: number, changed: Partial<any>) => {
            const acc = this.account(accountId);
            if (!acc) {
                return;
            }
            for (const key of Object.keys(changed)) {
                acc[key] = changed[key];
            }
            this.onRateChange();
        });

        this.socket.on('new', (account: any) => {
            if (isEmpty(account) || isEmpty(account.id)) {
                return;
            }

            const acc = this.account(account.id);
            if (acc) {
                return;
            }

            this.addOrUpdate(account);
            this.onRateChange();
        });
    }

    private startLoadData(): void {
        this.socket.once('dataR', (data: any[]) => {
            if (!data) {
                this.eventSubject.next({ name: 'data_loaded', args: [this.accounts] });
                return;
            }

            for (const acc of data) {
                this.addOrUpdate(acc);
            }

            this.onRateChange();
            this.eventSubject.next({ name: 'data_loaded', args: [this.accounts] });
        });

        this.socket.emit('data');
    }

    protected onConnect(): void {
        this.startLoadData();
    }

    protected onClear(): void {
        super.onClear();
        this.accounts.length = 0;
    }

    private onRateChange(): void {
        if (this.accounts.length < 1) {
            return;
        }

        // get the total in default currency
        let totalEur = new DecimalJS(0);
        let totalUsd = new DecimalJS(0);

        // group accounts
        for (const account of this.accounts) {
            totalEur = totalEur.plus(account.rate());
            totalUsd = totalUsd.plus(account.rate('USD'));
        }

        this.totals.EUR = totalEur.toNumber();
        this.totals.USD = totalUsd.toNumber();
    }

    private addOrUpdate(acc: any): void {
        let newAccount = false;
        let existing = this.accounts.find(c => c.id === acc.id);

        if (!existing) {
            existing = new Account(this);
            this.accounts.push(existing);
            newAccount = true;
        }

        existing.id = acc.id;
        existing.type = acc.type;
        existing.currencyCode = acc.currencyCode;
        existing.balance = acc.balance;
        existing.type = acc.type;
        existing.label = acc.label;
        existing.details = acc.details;

        if (newAccount) {
            this.newAccountAdded.emit(existing);
        }
    }

    // other listeners
    private initOutListeners(): void {
        this.currencyService.onRateChange(this.onRateChange.bind(this));
    }
}
