import { Component, OnInit, Input, OnDestroy } from '@angular/core';
import { MediaObserver } from '@angular/flex-layout';
import { AccountTableData } from '@app/pages/wallets/WalletsComponent';
import { MatDialog } from '@angular/material/dialog';
import { DialogComponent } from '../../dialog-component/DialogComponent';
import { trigger, state, animate, transition, style } from '@angular/animations';
import { Router } from '@angular/router';
import { Observable } from 'rxjs';
import { ExchangeRatesService } from '@app/shared/services/ExchangeRatesService';
import { Logger } from '@app/core/LoggerService';
import { SystemEventBus } from '@app/shared/event-bus/system.event.bus';

@Component({
  selector: 'app-account-table',
  templateUrl: 'accounts-table-component.html',
  styleUrls: ['./accounts-table-component.scss'],
  animations: [
    trigger('detailExpand', [
      state('collapsed', style({ height: '0px', minHeight: '0' })),
      state('expanded', style({ height: '*' })),
      transition('expanded <=> collapsed', animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')),
    ]),
  ],
})
export class AccountsTableComponent implements OnInit, OnDestroy {
  private readonly logger = new Logger(AccountsTableComponent.name);
  private eventListener: (...args: any[]) => void;

  public displayedColumns: string[];
  public expandedElement: AccountTableData | null;

  @Input() public dataSource: any;
  @Input() public tableType: string;
  @Input() public hideBalanceValue = true;
  @Input() public hideBalancePlaceholder = true;

  constructor(
    private readonly media: MediaObserver,
    private readonly dialog: MatDialog,
    private readonly router: Router,
    private readonly exchangeRatesService: ExchangeRatesService,
    private readonly systemEventBus: SystemEventBus,
  ) {}

  ngOnInit(): void {
    this.displayedColumns = this.handleTableType(this.tableType);

    this.eventListener = (tag, val) => {
      this.hideBalanceValue = val;
      this.hideBalancePlaceholder = !this.hideBalanceValue;
    };
    this.systemEventBus.addListener('system_settings_changed', this.eventListener);
  }

  ngOnDestroy(): void {
    this.systemEventBus.removeListener('system_settings_changed', this.eventListener);
  }

  get isMobile(): boolean {
    return this.media.isActive('xs') || this.media.isActive('sm');
  }

  public handleTableType(tableType: string): string[] {
    switch (tableType) {
      case 'wallet':
        return ['icon', 'accountName', 'assetType', 'amount', 'euro', 'receive', 'send', 'settings'];
      case 'vault':
        return ['icon', 'assetType', 'accountName', 'amount', 'euro', 'status', 'receive', 'send', 'settings'];
      default:
        return ['icon'];
    }
  }

  public isValidAccount(accountId: number): boolean {
    const account = this.dataSource.find((x) => x.id === accountId);
    if (account.hasOwnProperty('approved') && account.approved === false) {
      return false;
    }
    return true;
  }

  public expandMenu(element: any): void {
    if (this.isMobile) {
      return (this.expandedElement = this.expandedElement === element ? null : element);
    }
  }

  public openAccount(account: any): void {
    this.router.navigateByUrl(`/wallets/wallet-settings/${account.id}`);
  }

  public receive(account: any): void {
    //
    // TODO: Add correct walletAddress
    //
    const dialogRef = this.dialog.open(DialogComponent, {
      data: {
        type: 'receiveAmount',
        header: account.label,
        content: {
          currencyCode: account.currency.code,
          accountAddress: '0x0000000000000000000000000000000000000000',
        },
      },
      autoFocus: false
    });
  }

  public send(walletId: number): void {
    // TODO: What if zero?
    if (!walletId) {
      return;
    }
    this.router.navigateByUrl(`/wallets/create-transaction/${walletId}`);
  }

  public getAmountEuro(wallet: any): Observable<number> {
    const calcAmountObservable = this.exchangeRatesService.getExchangeValueObservable(wallet.currency.code, 'EUR', wallet.balance);
    return calcAmountObservable;
  }
}
