import { ExchangeRatesService } from '@app/shared/services/ExchangeRatesService';
import { NetworkFeeType } from './../../types/PaymentDetails';
import { ConfirmPayment } from './../../types/ConfirmPayment';
import { NetworkFeeLabelPipe } from './../../pipes/NetworkFeeLabelPipe';
import { Router } from '@angular/router';
import { SwalService } from '@app/shared/services/SwalService';
import { Component, Inject, OnInit } from '@angular/core';
import { MatDialogRef, MAT_DIALOG_DATA } from '@angular/material/dialog';
import { HttpClient } from '@angular/common/http';

interface ConfirmParams {
  title?: string;
  properties?: {
    label: string;
    value: string | number
  }[];
  formData?: {
    walletId: number;
    address: string;
    amount: number;
    note: string;
    networkFeeType: NetworkFeeType
  };
}

interface DialogData {
  // tslint:disable-next-line: ban-types
  [paymentType: string]: { buildParams: Function, apiCallUrl: string };
}

@Component({
  selector: 'app-confirm-payment-dialog',
  templateUrl: './confirm-payment-dialog.html',
  styleUrls: ['./confirm-payment-dialog.scss'],
})
export class ConfirmPaymentDialogComponent implements OnInit {
  public confirmData: ConfirmParams = {};

  private readonly dialogData: DialogData = {
    WITHDRAW: { buildParams: this.buildCryptoWithdrawParams.bind(this), apiCallUrl: '/payment/withdraw' }
  };

  constructor(
    @Inject(MAT_DIALOG_DATA) private readonly payment: ConfirmPayment,
    private readonly dialog: MatDialogRef<ConfirmPaymentDialogComponent>,
    private readonly httpClient: HttpClient,
    private readonly swalService: SwalService,
    private readonly router: Router,
    private readonly networkFeeLabelPipe: NetworkFeeLabelPipe,
    private readonly exchangeRatesService: ExchangeRatesService
  ) { }

  ngOnInit(): void {
    this.dialogData[this.payment.paymentType].buildParams();
  }

  private async buildCryptoWithdrawParams(): Promise<void> {
    this.confirmData = {
      title: 'Withdraw confirm',
      properties: [
        { label: 'Currency code', value: this.payment.currencyCode },
        { label: 'Amount', value: this.payment.amount },
        { label: 'Network fee', value: this.networkFeeLabelPipe.transform(this.payment.networkFeeType) },
        { label: 'Transaction fee', value: this.payment.paymentFee },
        { label: 'Total amount charged', value: this.payment.total },
        {
          label: 'Total amount charged EUR',
          value: await this.exchangeRatesService.getExchangeValueAsync(this.payment.currencyCode, 'EUR', this.payment.total)
        },
        { label: 'Recipient address', value: this.payment.address },
        { label: 'Note', value: this.payment.note }
      ],
      formData: {
        walletId: this.payment.walletId,
        address: this.payment.address,
        amount: this.payment.amount,
        note: this.payment.note,
        networkFeeType: this.payment.networkFeeType
      }
    };
  }

  public confirmPayment(): void {
    this.httpClient.post(this.dialogData[this.payment.paymentType].apiCallUrl, this.confirmData.formData).subscribe(data => {
      this.swalService.showSuccess({ title: 'Success!', text: 'Your order has been placed' })
        .then(() => {
          this.close();
          this.router.navigate(['/transactions']);
        });
    }, error => {
      this.swalService.showError({ title: 'Error', text: error.message });
    });
  }

  public close(): void {
    this.dialog.close();
  }
}
