import { HttpClient } from '@angular/common/http';
import { Component, OnInit, ViewChild } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { MatDialog } from '@angular/material/dialog';
import { ActivatedRoute } from '@angular/router';
import { Logger } from '@app/core/LoggerService';
import { DialogComponent } from '@app/shared/components/dialog-component/DialogComponent';
import { QrScannerDialogComponent } from '@app/shared/components/qr-scanner/QrScannerComponent';
import { SelectType } from '@app/shared/components/select-component/SelectComponent';
import { ExchangeRatesService } from '@app/shared/services/ExchangeRatesService';
import { FormFieldDesign } from '@app/shared/types/FormFieldDesign';
import { PaymentType } from '@app/shared/types/Payment';
import { CustomValidators } from '@app/shared/validators/CustomValidators';
import { ZXingScannerComponent } from '@zxing/ngx-scanner';
import { Decimal as DecimalJS } from 'decimal.js';
import { ConfirmPaymentDialogComponent } from './../../../shared/components/confirm-payment-dialog/ConfirmPaymentDialogComponent';
import { ConfirmPayment } from './../../../shared/types/ConfirmPayment';

export interface Account {
    id?: number;
    currencyCode?: string;
    label?: string;
    balance?: number;
    balanceEur?: number;
    currency?: {
        code: string,
    };
}

@Component({
    selector: 'app-create-transaction',
    templateUrl: './create-transaction-component.html',
    styleUrls: ['./create-transaction-component.scss'],
})
export class CreateTransactionComponent implements OnInit {
    private readonly logger = new Logger(CreateTransactionComponent.name);

    public formGroup: FormGroup;
    public account: Account = null;

    // TODO: Naming, see 'networkFee'
    private networkFees: { [type: string]: number } = {
        SOON_AS_POSSIBLE: 0,
        REGULAR: 0
    };

    public formFields = [
        { label: 'To:', formControlName: 'recipient' },
        { label: 'Note:', formControlName: 'note' },
        { label: 'Amount:', formControlName: 'amount', suffix: { type: 'text', text: '' } },
        { label: 'Amount in EUR:', formControlName: 'amountEUR', suffix: { type: 'text', text: 'EUR' } },
        { label: 'NXACustody Fee:', formControlName: 'internalFee', suffix: { type: 'text', text: '' } },
        { label: 'Total:', formControlName: 'totalAmount', suffix: { type: 'text', text: '' } },
    ];

    // TODO: Naming, see 'networkFees'
    public networkFee: SelectType = {
        header: 'Network Fee:',
        bold: true,
        options: []
    };

    public design: FormFieldDesign = {
        header: 'no-bold',
        field: 'stroked',
    };

    public requiredSigners: string[] = ['Eduards@test.test', 'Krisjanis@test.test'];

    @ViewChild('scanner') scanner: ZXingScannerComponent;

    constructor(
        private readonly formBuilder: FormBuilder,
        private readonly exchangeRatesService: ExchangeRatesService,
        private readonly httpClient: HttpClient,
        private readonly activatedRoute: ActivatedRoute,
        private readonly matDialog: MatDialog
    ) {
        this.formGroup = this.formBuilder.group({
            recipient: [null, Validators.required],
            amount: [0, Validators.required],
            note: [],
            amountEUR: [0],
            internalFee: [],
            totalAmount: [0],
            networkFeeType: ['REGULAR', Validators.required]
        });
    }

    async ngOnInit(): Promise<void> {
        const walletId = Number(this.activatedRoute.snapshot.params.id);
        // TODO: Why 'any'?
        this.httpClient.get<any>(`/wallet/${walletId}`).subscribe(async data => {
            this.account = data;
            this.account.currencyCode = data.currency.code;
            this.account.balanceEur = await this.exchangeRatesService
                .getExchangeValueAsync(this.account.currencyCode, 'EUR', this.account.balance);

            this.formFields[2].suffix.text = data.currency.code;
            this.formFields[4].suffix.text = data.currency.code;
            this.formFields[5].suffix.text = data.currency.code;

            this.formGroup.get('totalAmount').setValidators([CustomValidators.enoughFunds(this.account.balance)]);
            await this.setTransactionFees();
        });
    }

    private async setTransactionFees(): Promise<void> {
        // TODO: API call to DB to get internal transaction fee
        const transactionFees: { soonAsPosible: number, regular: number, internalFee: number } = {
            soonAsPosible: 0.0002,
            regular: 0.0001,
            internalFee: 0.08
        };

        this.setNetworkFee(transactionFees.regular, transactionFees.soonAsPosible);

        const internallFee = await this.exchangeRatesService
            .getExchangeValueAsync('EUR', this.account.currencyCode, transactionFees.internalFee);
        this.formGroup.get('internalFee').setValue(internallFee);

        this.calculateTotal();
    }

    private setNetworkFee(regular: number, soonAsPosible: number): void {
        this.networkFees.REGULAR = regular;
        this.networkFees.SOON_AS_POSSIBLE = soonAsPosible;

        // TODO: Why 'push' not assign?
        this.networkFee.options.push(
            {
                value: 'REGULAR',
                viewValue: `Regular (${regular} ${this.account.currencyCode})`
            },
            {
                value: 'SOON_AS_POSSIBLE',
                viewValue: `Soon as possible (${soonAsPosible} ${this.account.currencyCode})`
            }
        );
    }

    private getNetworkFeeAmount(type: string): number {
        return this.networkFees[type];
    }

    public openScanner(currencyCode: string): void {
        this.matDialog
            .open(QrScannerDialogComponent, {
                data: { currencyCode },
                panelClass: 'scannerDialog',
                maxWidth: '100vw',
                maxHeight: '100vh',
                height: '100%',
                width: '100%',
            })
            .afterClosed()
            .subscribe(async (result) => {
                if (result) {
                    this.formGroup.get('recipient').setValue(result.address);
                    if (result.amount) {
                        this.formGroup.get('amount').setValue(result.amount);
                        await this.updateExchangeRate(result.amount);
                    }
                    if (result.note) {
                        this.formGroup.get('note').setValue(result.note);
                    }
                }
            });
    }

    public openAddressBook(): void {
        this.matDialog.open(DialogComponent, {
            data: {
                type: 'addressBook',
                header: 'Address book',
            },
            width: '45rem',
        });
    }

    public async updateExchangeRate(value: string, eur?: boolean): Promise<void> {
        if (!value) {
            this.formGroup.get('amount').setValue(null);
            this.formGroup.get('amountEUR').setValue(0);
            this.formGroup.get('totalAmount').setValue(0);
            return;
        }

        // TODO: Why 'if'?
        if (value.indexOf(',')) {
            value = value.replace(',', '.');
        }

        if (eur) {
            this.formGroup.get('amountEUR').setValue(value);
            this.formGroup.get('amount').setValue(
                await this.exchangeRatesService.getExchangeValueAsync('EUR', this.account.currencyCode, value)
            );
        } else {
            this.formGroup.get('amountEUR').setValue(
                await this.exchangeRatesService.getExchangeValueAsync(this.account.currencyCode, 'EUR', value)
            );
            this.formGroup.get('amount').setValue(value);
        }

        this.calculateTotal();
    }

    public calculateTotal(): void {
        if (this.formGroup.get('amount').value === null) {
            return;
        }

        const internallFee = this.formGroup.get('internalFee').value;
        const networkFee = this.getNetworkFeeAmount(this.formGroup.get('networkFeeType').value);
        const calculatedTotal = new DecimalJS(this.formGroup.get('amount').value).add(internallFee).add(networkFee);
        this.formGroup.get('totalAmount').setValue(calculatedTotal.toNumber());
    }

    public submit(): void {
        if (this.formGroup.invalid) {
            return;
        }

        const networkFeeAmount = this.getNetworkFeeAmount(this.formGroup.get('networkFeeType').value);

        const formData: ConfirmPayment = {
            walletId: this.account.id,
            paymentType: PaymentType.WITHDRAW,
            currencyCode: this.account.currencyCode,
            amount: Number(this.formGroup.get('amount').value),
            address: this.formGroup.get('recipient').value,
            note: this.formGroup.get('note').value,
            networkFeeType: this.formGroup.get('networkFeeType').value,
            total: this.formGroup.get('totalAmount').value,
            paymentFee: new DecimalJS(this.formGroup.get('internalFee').value).add(networkFeeAmount).toNumber()
        };

        this.matDialog.open(ConfirmPaymentDialogComponent, {
            data: formData,
            width: '55rem'
        });
    }
}
