import { Component, OnDestroy, OnInit } from '@angular/core';
import { Logger } from '@app/core/LoggerService';
import { MatDialog } from '@angular/material/dialog';
import { DialogComponent } from '@app/shared/components/dialog-component/DialogComponent';
import { HttpClient } from '@angular/common/http';
import { ExchangeRatesService } from '@app/shared/services/ExchangeRatesService';
import { SystemEventBus } from '@app/shared/event-bus/system.event.bus';
import { AssetTypes } from '@app/shared/constant/AssetTypes';
import { WalletTypes } from '@app/shared/constant/WalletTypes';
import Decimal from 'decimal.js';
import { Observable } from 'rxjs';

export interface AccountTableData {
  currencyCode: string;
  accountName: string;
  amount: number;
  euro: number;
  accountId: number;
  settingCode1: string;
  settingCode2: string;
  expandText?: string;
}

export interface PendingTableData {
  currencyCode: string;
  accountName: string;
  accountId: number;
  settingType: string;
}

const pendingSource: PendingTableData[] = [
  {
    currencyCode: 'BTC',
    accountName: 'Wages',
    accountId: 1432,
    settingType: 'confirm'
  },
  {
    currencyCode: 'DASH',
    accountName: 'Cold',
    accountId: 1433,
    settingType: 'approve'
  }
];

@Component({
  selector: 'app-wallets',
  templateUrl: './wallets-component.html',
  styleUrls: ['./wallets-component.scss'],
})
export class WalletsComponent implements OnInit, OnDestroy {
  private readonly logger = new Logger(WalletsComponent.name);
  private eventListener: (...args: any[]) => void;

  public readonly accountList = WalletTypes;
  public readonly assetTypes = AssetTypes;

  public selectedFilterAccount = this.accountList[0].value;
  public selectedFilterAsset = this.assetTypes[0].value;
  public totalBalance = 0.0;
  public hideBalanceValue = true;
  public hideBalancePlaceholder = true;
  public dataSourcePending = pendingSource;
  public dataSource: any;

  constructor(
    private readonly dialog: MatDialog,
    private readonly httpClient: HttpClient,
    private readonly exchangeRatesService: ExchangeRatesService,
    private readonly systemEventBus: SystemEventBus
  ) {
  }

  async ngOnInit(): Promise<void> {
    this.eventListener = (tag, val) => {
      this.hideBalanceValue = val;
      this.hideBalancePlaceholder = !this.hideBalanceValue;
    };
    this.systemEventBus.addListener('system_settings_changed', this.eventListener);

    this.refresh();
  }

  ngOnDestroy(): void {
    this.systemEventBus.removeListener('system_settings_changed', this.eventListener);
  }

  public refresh(): void {
    this.receiveHideBalance();
    this.receiveWallets();
  }

  public receiveWallets(): void {
    this.httpClient.get('/wallet').subscribe(async (wallets: any[]) => {
      wallets = wallets.map(wallet => {
        wallet.balance = +wallet.balance;
        return wallet;
      });
      this.dataSource = wallets;

      const totalBalanceEur = await wallets.reduce(async (accumulatedPromise, wallet): Promise<Decimal> => {
        const inEur = await this.exchangeRatesService.getExchangeValueAsync(wallet.currency.code, 'EUR', wallet.balance);
        const accumulated = await accumulatedPromise;
        this.logger.debug(`>>>>> ${wallet.currency.code} : ${wallet.balance} : ${inEur} : ${accumulated}`);
        return accumulated.add(inEur);
      }, new Decimal(0));

      this.totalBalance = totalBalanceEur.toNumber();
    });
  }

  public receiveHideBalance(): void {
    this.httpClient.get<boolean>('/user/settings/hide-balances').subscribe(
      (hideBalance) => {
        this.hideBalanceValue = hideBalance;
        this.hideBalancePlaceholder = !this.hideBalanceValue;
      },
      (err) => {
        this.logger.error(`Error: ${err.message || err}`);
      }
    );
  }

  public createWallet(): void {
    const dialogRef = this.dialog.open(DialogComponent, {
      data: {
        type: 'createWallet',
        header: 'Create wallet'
      },
      width: '45rem',
    });
    dialogRef.afterClosed().subscribe(res => {
      if (res === 'walletSuccess'){
        this.refresh();
      }
    });
  }

  private calculateTotalWalletsBalance(wallets: any[]): Observable<number> {
    return new Observable(observer => {

    });
  }
}
