import { Component, OnDestroy, OnInit } from '@angular/core';
import { Logger } from '@app/core/LoggerService';
import { trigger, state, style, animate, transition } from '@angular/animations';
import { MediaObserver } from '@angular/flex-layout';
import { MatDialog } from '@angular/material/dialog';
import { DialogComponent } from '@app/shared/components/dialog-component/DialogComponent';
import { HttpClient } from '@angular/common/http';
import { ExchangeRatesService } from '@app/shared/services/ExchangeRatesService';
import { SystemEventBus } from '@app/shared/event-bus/system.event.bus';

export interface TableData {
    currencyCode: string;
    vaultName: string;
    amount: number;
    euro: number;
    accountId: number;
    settingCode1: string;
    settingCode2: string;
    expandText?: string;
}

const tableSource: any[] = [
    {
        assetType: 'crypto',
        balance: 0,
        balanceFrozen: '0.000000000000000000',
        balanceOnHold: '0.000000000000000000',
        currency: { code: 'ETH', codeLabel: 'ETH', symbol: 'ETH', name: 'Ethereum', type: 'crypto' },
        description: '',
        disabled: false,
        id: 2,
        label: 'ETH Wallet',
        totalReceived: '0.000000000000000000',
        totalSent: '0.000000000000000000',
        uuid: 'b048e425-d594-4297-bb6e-add18f3f7857',
        walletType: 'private',
        approved: false,
        accountType: 'vault'
    },
    {
        assetType: 'crypto',
        balance: 0,
        balanceFrozen: '0.000000000000000000',
        balanceOnHold: '0.000000000000000000',
        currency: { code: 'ETH', codeLabel: 'ETH', symbol: 'ETH', name: 'Ethereum', type: 'crypto' },
        description: '',
        disabled: false,
        id: 3,
        label: 'ETH Wallet',
        totalReceived: '0.000000000000000000',
        totalSent: '0.000000000000000000',
        uuid: 'b048e425-d594-4297-bb6e-add18f3f7857',
        walletType: 'private',
        approved: true,
        accountType: 'vault'
    }
];

@Component({
    selector: 'app-vault',
    templateUrl: './vault-component.html',
    styleUrls: ['./vault-component.scss'],
    animations: [
        trigger('detailExpand', [
            state('collapsed', style({ height: '0px', minHeight: '0' })),
            state('expanded', style({ height: '*' })),
            transition('expanded <=> collapsed', animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')),
        ]),
    ],
})
export class VaultComponent implements OnInit, OnDestroy {
    private readonly logger = new Logger(VaultComponent.name);
    private eventListener: (...args: any[]) => void;

    public hideBalanceValue = true;
    public hideBalancePlaceholder = true;

    public accountList = [
        { viewValue: 'All Vaults', value: 'all' },
        { viewValue: 'Private (Code A:11)', value: 'private' },
        { viewValue: 'Business (Code A:21)', value: 'business' },
        { viewValue: 'Asset Manager (Code A:31)', value: 'manager' },
    ];

    public assetTypes = [
        { viewValue: 'All Asset Types', value: 'all' },
        { viewValue: 'Digital Fiat (Code T:1)', value: 'digital' },
        { viewValue: 'Security tokens (Code T:2)', value: 'security' },
        { viewValue: 'Crypto Currency (Code T:3)', value: 'crypto' },
        { viewValue: 'Property Backed Tokens (Code T:4)', value: 'property' },
        { viewValue: 'Digital Collectibles (Code T:5)', value: 'collectible' },
    ];

    public createVaultInfo = {
        security: {
            icon: 'shield',
            text: 'Your account will have banking-level security<br>standards and secure tools for digital asset protection',
            features: [
                'Offline NXACustody signature',
                'Cold storage (keys)',
                'Confirmation call from NXACustody team for all transactions above 100 000 EUR'
            ]
        },
        pricing: {
            icon: 'pricing',
            text: {
                pricing: 'Vault ownership pricing',
                request: 'Request special rate for accounts starting 1 000 000 EUR and above'
            },
            features: [
                { text: 'Transaction', percentage: 0.75 },
                { text: 'Balance maintenance fee per month', percentage: 0.075 }
            ]
        }
    };
    public expandedElement: TableData | null;
    public totalBalance = 0.0;
    public dataSource: string[] = [];
    public selectedFilterAccount = this.accountList[0].value;
    public selectedFilterAsset = this.assetTypes[0].value;

    constructor(
        private readonly media: MediaObserver,
        private readonly dialog: MatDialog,
        private readonly httpClient: HttpClient,
        private readonly exchangeRatesService: ExchangeRatesService,
        private readonly systemEventBus: SystemEventBus
    ) {
    }

    ngOnInit(): void {
        this.eventListener = (tag, val) => {
            this.hideBalanceValue = val;
            this.hideBalancePlaceholder = !this.hideBalanceValue;
        };
        this.systemEventBus.addListener('system_settings_changed', this.eventListener);

        this.receiveHideBalance();

        // TODO: DISABLED for Demo only:
        // this.refresh();
    }

    ngOnDestroy(): void {
        this.systemEventBus.removeListener('system_settings_changed', this.eventListener);
    }

    expandMenu(element: any): void {
        if (this.isMobile) {
            return (this.expandedElement = this.expandedElement === element ? null : element);
        }
    }

    public receive(account: any): void {
        const dialogRef = this.dialog.open(DialogComponent, {
            data: {
                type: 'receiveAmount',
                header: 'Vault Address',
                content: {
                    accountAddress: '235346545643434',
                },
            },
            width: '45rem',
        });
    }

    get isMobile(): boolean {
        return this.media.isActive('xs') || this.media.isActive('sm');
    }

    public createVault(): void {
        const dialogRef = this.dialog.open(DialogComponent, {
            data: {
                type: 'createVault',
                header: 'Vault creation request'
            },
            width: '45rem',
        });

        dialogRef.afterClosed().subscribe(res => {
            if (res === 'vaultSuccess') {
                this.refresh();
            }
        });
    }

    public refresh(): void {

        this.dataSource = tableSource;

        // this.httpClient.get('/vault').subscribe((vaults: any[]) => {
        //   this.hasVault = Array.isArray(vaults) && vaults.length > 0 ? true : false;
        //   vaults = vaults.map(vaults => {
        //     vaults.balance = +vaults.balance;
        //     return vaults;
        //   });
        //   this.dataSource = vaults;
        // });
    }

    public receiveHideBalance(): void {
        this.httpClient.get<boolean>('/user/settings/hide-balances').subscribe(
            (hideBalance) => {
                this.hideBalanceValue = hideBalance;
                this.hideBalancePlaceholder = !this.hideBalanceValue;
            },
            (err) => {
                this.logger.error(`Error: ${err.message || err}`);
            }
        );
    }
}
