import { Component, OnDestroy, OnInit } from '@angular/core';
import { FormBuilder, Validators, FormGroup } from '@angular/forms';
import { CustomValidators } from '../../shared/validators/CustomValidators';
import { SwalService } from '../../shared/services/SwalService';
import { HttpClient } from '@angular/common/http';
import { UserService } from '../../shared/services/UserService';
import { ERROR } from '../../shared/Errors';
import { DialogComponent } from '../../shared/components/dialog-component/DialogComponent';
import { MatDialog } from '@angular/material/dialog';
import { FormFieldDesign } from '@app/shared/types/FormFieldDesign';
import { Logger } from '@app/core/LoggerService';
import { SystemEventBus } from '@app/shared/event-bus/system.event.bus';

@Component({
  selector: 'app-security',
  templateUrl: './security-component.html',
  styleUrls: ['./security-component.scss'],
})
export class SecurityComponent implements OnInit, OnDestroy {
  private readonly logger = new Logger(SecurityComponent.name);

  private eventListener: (...args: any[]) => void;

  public passwordForm: FormGroup;

  public changePassFields = [
    {label: 'Current password', type: 'password', formControlName: 'currentPassword'},
    {label: 'New password', type: 'password', formControlName: 'newPassword'},
    {label: 'Confirm password', type: 'password', formControlName: 'passwordConfirm'}
  ];

  public design: FormFieldDesign = {
    header: 'no-bold',
    field: 'stroked'
  };

  // setting values
  public confirmCall = true;
  public hideBalanceValue: boolean;
  public withdrawsWhitelistEnabled: boolean;
  public twoFA: boolean;
  public recoveryEmail: boolean;
  public recoveryNumber: boolean;

  constructor(
    public readonly userService: UserService,
    private readonly formBuilder: FormBuilder,
    private readonly httpClient: HttpClient,
    private readonly swalService: SwalService,
    private readonly dialog: MatDialog,
    private readonly systemEventBus: SystemEventBus
  ) {
    const password = this.formBuilder.control('', [
      Validators.required,
      Validators.minLength(8),
      Validators.maxLength(14),
      Validators.pattern(/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/),
    ]);

    this.passwordForm = this.formBuilder.group({
      currentPassword: ['', Validators.required],
      newPassword: password,
      passwordConfirm: ['', CustomValidators.match(password)],
    });
  }

  ngOnInit(): void {
    this.refresh();

    this.eventListener = (evt, ...args) => this.onSystemSettingsChanged(evt, ...args);
    this.systemEventBus.addListener('system_settings_changed', this.eventListener);
  }

  ngOnDestroy(): void {
    this.systemEventBus.removeListener('system_settings_changed', this.eventListener);
  }

  refresh(): void {
    this.receiveHideBalance();
    this.receiveRestrictWithdrawsWhitelist();
  }

  //
  // Change Password
  //
  public async changePassword(): Promise<void> {
    const me = await this.httpClient.get<any>('/user/me').toPromise();
    const changePasswordPostData = {
      email: me.user.email,
      password: this.passwordForm.get('currentPassword').value,
      newPassword: this.passwordForm.get('newPassword').value
    };
    this.httpClient.post('/auth/change-password', changePasswordPostData).subscribe(
      (data) => {
        this.userService.reload();
        this.systemEventBus.triggerEvent('password_changed', 'change-password', me.user.email);
      },
      (err) => {
        this.swalService.showError({ title: 'Error', text: err.message });
      }
    );
  }

  //
  // Receive User's Hide Balance preference via API
  //
  public receiveHideBalance(): void {
    this.httpClient.get<boolean>('/user/settings/hide-balances').subscribe(
      (hideBalance) => {
        this.userService.reload();
        this.hideBalanceValue = hideBalance;
      },
      (err) => {
        this.swalService.showError({ title: 'Error', text: err.message });
      }
    );
  }

  //
  // Store User's Hide Balance preference via API
  //
  public toggleHideBalance(event: any): void {
    this.hideBalanceValue = event.checked;

    this.httpClient.put<boolean>('/user/settings/hide-balances', {on: this.hideBalanceValue}).subscribe(
      (hideBalance) => {
        this.hideBalanceValue = hideBalance;
        event.source.checked = this.hideBalanceValue;
        this.systemEventBus.triggerEvent('system_settings_changed', 'hide-balances', this.hideBalanceValue);
      },
      (err) => {
        this.receiveHideBalance();
        event.source.checked = this.hideBalanceValue;
        this.swalService.showError({ title: 'Error', text: err.message });
      },
      () => {
        event.source.checked = this.hideBalanceValue;
      }
    );
  }

  //
  // Receive Withdraws whitelist preference preference via API
  //
  private receiveRestrictWithdrawsWhitelist(): void {
    this.httpClient.get<boolean>('/user/settings/restrict-withdrawals-to-addressbook').subscribe(
      (withdrawsWhitelistEnabled) => {
        this.withdrawsWhitelistEnabled = withdrawsWhitelistEnabled;
      },
      (err) => {
        this.swalService.showError({ title: 'Error', text: err.message });
      }
    );
  }

  //
  // Store User's Withdraws whitelist preference via API
  //
  public changeRestrictWithdrawsWhitelist(event: any): void {
    this.withdrawsWhitelistEnabled = event.checked;

    this.httpClient.put<boolean>('/user/settings/restrict-withdrawals-to-addressbook', { on: this.withdrawsWhitelistEnabled })
      .subscribe((enabled) => {
          this.withdrawsWhitelistEnabled = enabled;
          event.source.checked = this.withdrawsWhitelistEnabled;
        },
        (err) => {
          this.receiveRestrictWithdrawsWhitelist();
          event.source.checked = this.withdrawsWhitelistEnabled;
          this.swalService.showException(err, [
            ERROR.BASIC_AUTH_WRONG_CREDINTALS_EMAIL,
            ERROR.BASIC_AUTH_WRONG_CREDINTALS_PWD,
          ]);
        },
        () => {
          //
          // FIX: PB-273 - Restrict withdrawals to the address book addresses only (e.g toggle must reflect real state)
          //
          event.source.checked = this.withdrawsWhitelistEnabled;
        }
      );
  }

  //
  // Enable/Disable 2FA
  //
  public change2FA(action: string): void {
    if (action === 'enable') {
      const dialogRef = this.dialog.open(DialogComponent, {
        data: {
          type: 'enable2FA',
          header: 'Two factor Authorization',
          content: {
            helper: 'Scan this QR code with your two factor authentication device to set up your account',
            qrCode: '2353465456',
          } },
        width: '45rem',
      });
    } else if (action === 'disable') {
      const dialogRef = this.dialog.open(DialogComponent, {
        data: { type: 'disable2FA' },
        width: '45rem',
      });
    }
  }

  //
  // Add/Edit recovery email
  //
  public changeRecoveryEmail(action: string): void {
    if (action === 'set'){
      const dialogRef = this.dialog.open(DialogComponent, {
        data: {
          type: 'addRecoveryEmail',
          header: 'Set recovery email' },
        width: '45rem',
      });
    } else if ( action === 'edit'){
      const dialogRef = this.dialog.open(DialogComponent, {
        data: {
          type: 'editRecoveryEmail',
          header: 'Edit recovery email' },
        width: '45rem',
      });
    }
  }

  //
  // Set/Edit recovery number
  //
  public changeRecoveryPhoneNumber(action: string): void {
    if (action === 'set'){
      const dialogRef = this.dialog.open(DialogComponent, {
        data: {
          type: 'addRecoveryPhoneNumber',
          header: 'Set recovery phone number' },
        width: '45rem',
      });
    } else if ( action === 'edit'){
      const dialogRef = this.dialog.open(DialogComponent, {
        data: {
          type: 'editRecoveryPhoneNumber',
          header: 'Edit recovery phone number' },
        width: '45rem',
      });
    }
  }

  private onSystemSettingsChanged(...args: any[]): void {
    this.refresh();
  }
}
