import { environment } from '@env/environment';
import { Component, AfterViewInit, OnDestroy } from '@angular/core';
import { Logger } from '@app/core/LoggerService';
import { FormGroup, FormBuilder, Validators } from '@angular/forms';
import { FormFieldDesign } from '@app/shared/types/FormFieldDesign';
import { HttpClient } from '@angular/common/http';
import { Router } from '@angular/router';
import { SwalService } from '@app/shared/services/SwalService';
import { MatDialog } from '@angular/material/dialog';
import { InviteCodeDialogComponent } from '@app/shared/components/invite-code-dialog/InviteCodeDialog';
import { CustomValidators } from '@app/shared/validators/CustomValidators';
import { DatePipe } from '@angular/common';
import { Subscription } from 'rxjs';
import * as countries from 'i18n-iso-countries';

@Component({
  selector: 'app-register',
  templateUrl: './register-component.html',
  styleUrls: ['./register-component.scss'],
})
export class RegisterComponent implements AfterViewInit, OnDestroy {
  private readonly logger = new Logger(RegisterComponent.name);

  private birthDateSubscription: Subscription;
  private countrySubscription: Subscription;
  public readonly landingUrl = environment.landingUrl;
  public formGroup: FormGroup;

  public design: FormFieldDesign = {
    header: 'bold',
    field: 'basic',
  };

  public fields = [
    { label: 'First name', formControlName: 'firstName' },
    { label: 'Last name', formControlName: 'lastName' },
    { label: 'Email', formControlName: 'email' },
    { label: 'Password', formControlName: 'password' },
    { label: 'Confirm password', formControlName: 'confirmPassword' },
    { label: 'Country', formControlName: 'country' },
    { label: 'Phone number', formControlName: 'phoneNumber', formControlNameCallCode: 'phoneNumberCallCode' },
    { label: 'Birth date', formControlName: 'birthDate' }
  ];

  constructor(
    private readonly formBuilder: FormBuilder,
    private readonly httpClient: HttpClient,
    private readonly router: Router,
    private readonly swalService: SwalService,
    private readonly dialog: MatDialog,
    private readonly datePipe: DatePipe,
  ) {
    const password = this.formBuilder.control(null, [
      Validators.required,
      Validators.minLength(8),
      Validators.maxLength(14),
      Validators.pattern(/^(?=.*[A-Z])(?=.*[0-9])(?=.{8,})/)
    ]);
    this.formGroup = this.formBuilder.group({
      firstName: [null, Validators.required],
      lastName: [null, Validators.required],
      email: [null, [Validators.required, Validators.email]],
      password,
      confirmPassword: [null, [Validators.required, CustomValidators.match(password)]],
      terms: [null, Validators.required],
      invitationCode: [null, Validators.required],
      politicalExposed: [null, Validators.required],
      isUsCitizen: [null, Validators.required],
      country: [null, Validators.required],
      phoneNumber: [null, Validators.required],
      phoneNumberCallCode: [null, Validators.required],
      birthDate: [null, [Validators.required]]
    });
  }

  ngAfterViewInit(): void {
    setTimeout(() => {
      this.prepareForm();
      this.formGroup.disable();
      this.openInviteDialog();
    }, 300);
  }

  ngOnDestroy(): void {
    if (this.birthDateSubscription) {
      this.birthDateSubscription.unsubscribe();
    }
    if (this.countrySubscription) {
      this.countrySubscription.unsubscribe();
    }
  }

  private prepareForm(): void {
    const minRequiredDate = new Date();
    const control = this.formGroup.controls;

    minRequiredDate.setFullYear(minRequiredDate.getFullYear() - 18);

    this.birthDateSubscription = control.birthDate.valueChanges.subscribe(() => {
      if (minRequiredDate < this.formGroup.get('birthDate').value) {
        this.formGroup.get('birthDate').setErrors({ tooYoung: true });
      } else {
        this.formGroup.get('birthDate').setErrors(null);
      }
    });

    this.countrySubscription = control.country.valueChanges.subscribe(() => {
      if (!countries.isValid(countries.getAlpha3Code(this.formGroup.get('country').value, 'en'))) {
        this.formGroup.get('country').setErrors({ invalidCountry: true });
      } else {
        this.formGroup.get('country').setErrors(null);
      }
    });
  }

  public openInviteDialog(): void {
    const dialogRef = this.dialog.open(InviteCodeDialogComponent, {
      disableClose: true,
      autoFocus: false,
      data: {
        type: 'inviteCode',
        header: 'Verify your invite',
      },
      width: '35rem',
    });
    dialogRef.afterClosed().subscribe((inviteCode: string) => {
      if (inviteCode) {
        this.formGroup.enable();
        this.formGroup.get('invitationCode').setValue(inviteCode);
      }
    });
  }

  public register(): void {
    if (this.formGroup.invalid) {
      return;
    }

    const birthDate = this.datePipe.transform(this.formGroup.get('birthDate').value, 'yyyy-MM-dd');
    const fullPhoneNumber: string = '+' + this.formGroup.get('phoneNumberCallCode').value + this.formGroup.get('phoneNumber').value;

    const formData = {
      invite: this.formGroup.get('invitationCode').value,
      email: this.formGroup.get('email').value,
      password: this.formGroup.get('password').value,
      firstName: this.formGroup.get('firstName').value,
      lastName: this.formGroup.get('lastName').value,
      isPep: this.formGroup.get('politicalExposed').value,
      isUSCitizen: this.formGroup.get('isUsCitizen').value,
      phoneNumber: fullPhoneNumber,
      birthDate,
      country: this.formGroup.get('country').value
    };

    this.httpClient
      .disableAuthToken()
      .post('/user/register', formData)
      .subscribe(
        (data: any) => {
          this.swalService.showSuccess({ title: 'Success', html: 'We\'ve sent an email to your email account to confirm your account.<br><br>Please check your email.' });
          this.router.navigate(['/login']);
        },
        (error) => {
          this.swalService.showError({ title: 'Error', text: error.message });
        }
      );

  }
}
