import { Component, OnInit, ViewChild } from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { Logger } from '@app/core/LoggerService';
import { CryptoMarketOhlc, CryptoMarketRate, ExchangeRatesService, OhlcInterval } from '@app/shared/services/ExchangeRatesService';
import { ChartDataSets, ChartOptions, ChartType } from 'chart.js';
import { Color, BaseChartDirective, Label } from 'ng2-charts';
import { DatePipe } from '@angular/common';

@Component({
  selector: 'app-market-pair-chart-component',
  templateUrl: './market-pair-chart-component.html',
  styleUrls: ['./market-pair-chart-component.scss'],
})
export class MarketPairChartComponent implements OnInit {
  private readonly logger = new Logger(MarketPairChartComponent.name);

  public currentRate: CryptoMarketRate = {
    pair: '',
    ticker: '',
    open: 0,
    average: 0,
    last: undefined,
    reversed: false,
    timestamp: 0,
    datetime: new Date()
  };

  public lineChartLegend = true;
  public lineChartType: ChartType = 'line';
  public lineChartLabels: Label[] = [];

  public lineChartData: ChartDataSets[] = [
    {
      data: [],
      label: ''
    }
  ];

  public lineChartOptions: (ChartOptions & { annotation: any }) = {
    responsive: true,
    scales: {
      xAxes: [{}],
      yAxes: [{}],
    },
    annotation: {
    },
  };

  public lineChartColors: Color[] = [
    // red
    {
      backgroundColor: 'rgba(255,0,0,0.0)',
      borderColor: 'rgba(255,0,0,0.5)',
      pointBackgroundColor: 'rgba(255,0,0,0.0)',
      pointBorderColor: 'rgba(255,0,0,0.0)',
      pointHoverBackgroundColor: 'rgba(255,0,0,0.5)',
      pointHoverBorderColor: 'rgba(255,0,0,0.5)'
    },
    // Blue
    {
      backgroundColor: 'rgba(0,0,255,0.0)',
      borderColor: 'rgba(0,0,255,0.5)',
      pointBackgroundColor: 'rgba(0,0,255,0.0)',
      pointBorderColor: 'rgba(0,0,255,0.0)',
      pointHoverBackgroundColor: 'rgba(0,0,255,0.5)',
      pointHoverBorderColor: 'rgba(0,0,255,0.5)'
    },
    // green
    {
      backgroundColor: 'rgba(0,255,0,0.0)',
      borderColor: 'rgba(0,255,0,0.5)',
      pointBackgroundColor: 'rgba(0,255,0,0.0)',
      pointBorderColor: 'rgba(0,255,0,0.0)',
      pointHoverBackgroundColor: 'rgba(0,255,0,0.5)',
      pointHoverBorderColor: 'rgba(0,255,0,0.5)'
    },
    // grey
    {
      backgroundColor: 'rgba(148,159,177,0.0)',
      borderColor: 'rgba(148,159,177,0.5)',
      pointBackgroundColor: 'rgba(148,159,177,0.0)',
      pointBorderColor: 'rgba(148,159,177,0.0)',
      pointHoverBackgroundColor: 'rgba(148,159,177,0.5)',
      pointHoverBorderColor: 'rgba(148,159,177,0.5)'
    },
    // dark grey
    {
      backgroundColor: 'rgba(188,199,207,0.0)',
      borderColor: 'rgba(188,199,207,0.5)',
      pointBackgroundColor: 'rgba(188,199,207,0.0)',
      pointBorderColor: 'rgba(188,199,207,0.0)',
      pointHoverBackgroundColor: 'rgba(188,199,207,0.5)',
      pointHoverBorderColor: 'rgba(188,199,207,0.5)'
    },
  ];

  @ViewChild(BaseChartDirective, { static: true }) chart: BaseChartDirective;

  constructor(
    private readonly route: ActivatedRoute,
    private readonly exchangeRatesService: ExchangeRatesService
  ) {
  }

  ngOnInit(): void {
    this.refresh();
  }

  refresh(): void {
    const base = this.route.snapshot.params.base;
    const quote = this.route.snapshot.params.quote;

    // Get Current
    this.exchangeRatesService.getExchangeRateObservable(base, quote).subscribe(
      (rate: CryptoMarketRate) => {
        this.logger.debug(`Got Exchange value for: ${base}${quote}: ${rate.datetime}`);
        this.currentRate = rate;
      },
      err => {
        this.logger.error(`Error: ${err.message || err}`);
      }
    );

    // Get OHLC
    this.exchangeRatesService.getOhlcObservable(base, quote, OhlcInterval.oneHour, this.currentRate.last).subscribe(
      (ohlc: CryptoMarketOhlc) => {
        this.logger.debug(`Got OHLC data for: ${ohlc.pair}`);

        const chartOpenData = ohlc.entries.map(e => parseFloat(e.open)).filter(e => e > 0);
        const chartOpenDataSet: ChartDataSets = {
          data: chartOpenData,
          label: 'Open',
          hidden: true
        };

        const chartHighData = ohlc.entries.map(e => parseFloat(e.high)).filter(e => e > 0);
        const chartHighDataSet: ChartDataSets = {
          data: chartHighData,
          label: 'High'
        };

        const chartLowData = ohlc.entries.map(e => parseFloat(e.low)).filter(e => e > 0);
        const chartLowDataSet: ChartDataSets = {
          data: chartLowData,
          label: 'Low'
        };

        const chartCloseData = ohlc.entries.map(e => parseFloat(e.close)).filter(e => e > 0);
        const chartCloseDataSet: ChartDataSets = {
          data: chartCloseData,
          label: 'Close',
          hidden: true
        };

        const chartVwapData = ohlc.entries.map(e => parseFloat(e.vwap)).filter(e => e > 0);
        const chartVwapDataSet: ChartDataSets = {
          data: chartVwapData,
          label: 'Vwap'
        };

        this.lineChartData = [
          chartHighDataSet,
          chartLowDataSet,
          chartVwapDataSet,
          chartOpenDataSet,
          chartCloseDataSet
        ];

        const datepipe: DatePipe = new DatePipe('en-US');

        this.lineChartLabels = ohlc.entries
          .filter(e => parseFloat(e.vwap) > 0)
          .map(e => {
            const date = new Date(e.time);
            const dateFormatted = datepipe.transform(date, 'HH:mm:ss');
            return dateFormatted;
          });
      },
      err => {
        this.logger.error(`Error: ${err.message || err}`);
      }
    );
  }

  // Handlers
  chartClicked({ event, active }: { event: MouseEvent, active: {}[] }): void {
    console.log(event, active);
  }

  chartHovered({ event, active }: { event: MouseEvent, active: {}[] }): void {
    console.log(event, active);
  }
}
