import { HttpClient } from '@angular/common/http';
import { Component, Input, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { Logger } from '@app/core/LoggerService';

@Component({
  selector: 'app-market',
  templateUrl: './market-component.html',
  styleUrls: ['./market-component.scss'],
})
export class MarketComponent implements OnInit {
  private readonly logger = new Logger(MarketComponent.name);

  public columns = 4;
  public rowHeight = '2:1.1';

  @Input() pairsMap = new Map<string, CryptoMarketRate>();

  constructor(
    private readonly route: ActivatedRoute,
    private readonly router: Router,
    private readonly httpClient: HttpClient
  ) {
  }

  ngOnInit(): void {
    this.columns = (window.innerWidth <= 1500) ? 1 : 4;
    this.rowHeight = (window.innerWidth <= 1500) ? '2:1.5' : '2:1.1';
    this.refresh();
  }

  onResize(event): void {
    this.columns = (event.target.innerWidth <= 1500) ? 1 : 4;
    this.rowHeight = (window.innerWidth <= 1500) ? '2:1.5' : '2:1.1';
  }

  refresh(): void {
    this.logger.debug(`Previous pairs: ${JSON.stringify(this.pairsMap)}`);

    this.httpClient.get<CryptoMarketRate[]>('/crypto-market-rates/pairs').subscribe(
      pairs => {
        this.pairsMap = pairs.reduce((acc, current) => {
            acc[current.pair] = current;
            return acc;
          },
          new Map<string, CryptoMarketRate>()
        );
        this.logger.debug(`Got pairs: ${JSON.stringify(this.pairsMap)}`);
      },
      err => {
        this.logger.error(`Error: ${err}`, err.stack);
      }
    );
  }

  openOhlcFor(base: string, quote): void {
    this.router.navigate([`ohlc-chart/${base}/${quote}`], { relativeTo: this.route, replaceUrl: true });
  }

  openHistory(ticker: string): void {
    this.router.navigate(['/transactions'], { replaceUrl: true });
  }
}

interface CryptoMarketRate {
  pair: string;
  ticker: string;
  open: string;
  average: string;
  last: string;
  timestamp: number;
}
