import { HttpClient, HttpErrorResponse } from '@angular/common/http';
import { Component, OnDestroy, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';
import { AuthenticationService } from '@app/core/authentication/AuthenticationService';
import { Logger } from '@app/core/LoggerService';
import { untilDestroyed } from '@app/core/UntilDestroyed';
import { EntryService } from '@app/shared/services/EntryService';
import { SwalService } from '@app/shared/services/SwalService';
import { FormFieldDesign } from '@app/shared/types/FormFieldDesign';
import { TranslateService } from '@ngx-translate/core';
import { detect } from 'detect-browser';
import * as Fingerprint2 from 'fingerprintjs2';
import { finalize } from 'rxjs/operators';
import Swal, { SweetAlertResult } from 'sweetalert2';

@Component({
  selector: 'app-login',
  templateUrl: './login-component.html',
  styleUrls: ['./login-component.scss'],
})
export class LoginComponent implements OnInit, OnDestroy {
  private readonly logger = new Logger(LoginComponent.name);

  public fields = [
    { label: 'Email', formControlName: 'email' },
    { label: 'Password', type: 'password', formControlName: 'password' },
  ];
  public design: FormFieldDesign = {
    header: 'no-bold',
    field: 'basic',
  };
  public error: string | undefined;
  loginForm!: FormGroup;
  isLoading = false;
  siteKey = '';
  hide = true;
  rememberMeCheckBox = false;

  private deviceInfo: {
    browser: string;
    platform: string;
    hash: string;
  } = null;

  constructor(
    private readonly router: Router,
    private readonly route: ActivatedRoute,
    private readonly formBuilder: FormBuilder,
    private readonly authenticationService: AuthenticationService,
    private readonly activatedRoute: ActivatedRoute,
    private readonly swalService: SwalService,
    private readonly entryService: EntryService,
    private readonly translateService: TranslateService,
    private readonly httpClient: HttpClient
  ) {
    this.loginForm = this.formBuilder.group({
      email: ['', [Validators.required, Validators.email]],
      password: ['', Validators.required],
      rememberMe: [null],
      // recaptchaReactive: ['', Validators.required]
    });
  }

  async ngOnInit(): Promise<void> {
    this.authenticationService.logout().subscribe((res) => {});

    Fingerprint2.get(
      {
        excludes: {
          adBlock: true,
          userAgent: true,
          language: true,
          colorDepth: true,
          deviceMemory: true,
          pixelRatio: true,
          hardwareConcurrency: true,
          screenResolution: true,
          availableScreenResolution: true,
          timezoneOffset: true,
          timezone: true,
          sessionStorage: true,
          localStorage: true,
          indexedDb: true,
          addBehavior: true,
          openDatabase: true,
          cpuClass: true,
          // platform: true,
          doNotTrack: true,
          plugins: true,
          // canvas: true,
          webgl: true,
          webglVendorAndRenderer: true,
          hasLiedLanguages: true,
          hasLiedResolution: true,
          hasLiedOs: true,
          hasLiedBrowser: true,
          touchSupport: true,
          // fonts: true,
          fontsFlash: true,
          audio: true,
          enumerateDevices: true,
        },
      },
      (components: any) => {
        const browser = detect();
        const values = components.map((c: any) => c.value);
        const fingerprintHash = Fingerprint2.x64hash128(values.join(''), 31);

        this.deviceInfo = {
          browser: browser.name,
          platform: browser.os,
          hash: fingerprintHash,
        };
      }
    );
  }

  ngOnDestroy(): void {
    // Used by 'untilDestroyed'
  }

  login(): void {
    if (this.loginForm.invalid) {
      return;
    }

    this.isLoading = true;
    this.error = null;
    this.authenticationService
      .login({
        email: this.loginForm.get('email').value,
        password: this.loginForm.get('password').value,
        // deviceInfo: this.deviceInfo,
      })
      .pipe(
        finalize(() => {
          this.loginForm.markAsPristine();
          this.isLoading = false;
        }),
        untilDestroyed(this)
      )
      .subscribe(
        (credentials: any) => {
          this.logger.debug(`${credentials.email} successfully logged in`);
          this.router.navigate([this.route.snapshot.queryParams.redirect || '/'], { replaceUrl: true });
        },
        (error: HttpErrorResponse) => {
          // TODO: Move to a separate method to handle this case and document it
          if (error.status === 403) {
            Swal.fire({
              position: 'center',
              title: 'Confirm your email',
              html: 'Please confirm your email before continuing.',
              showCancelButton: true,
              confirmButtonText: 'RESEND EMAIL',
              cancelButtonText: 'CLOSE'
            }).then((result: SweetAlertResult) => {

              if (!result.value || result.value !== true) {
                return;
              }

              const formData = {
                email: this.loginForm.get('email').value
              };

              this.httpClient.disableAuthToken().post('/user/email/confirmation/resend', formData).subscribe(data => {
                this.swalService.showSuccess({ title: 'Success', text: 'Email has been resent.'});
              },
              (err: HttpErrorResponse) => {
                this.logger.error(`Resend email error: ${err.message || err.statusText}`, err);
              });
            });
          } else {
            this.logger.error(`Login error: ${error.message || error.statusText}`, error);
            this.error = error.message || error.statusText;
          }
        }
      );
  }

  resolved(token: any): void {
    // console.log(token);
  }

  register(): void {
    if (!this.entryService.data || !this.entryService.data.registrationEnabled) {
      this.swalService.showInfo({
        title: 'SWAL.REGISTRATION_DISABLED_TITLE',
        text: 'SWAL.REGISTRATION_DISABLED_TEXT',
        textParams: { param_value: this.entryService.data.registrationDisabledReason, timer: null },
      });
      return;
    }

    if (this.entryService.data.regionRestricted) {
      this.swalService.showInfo({ title: 'SWAL.REGION_RESTRICTED_TITLE', text: 'SWAL.REGION_RESTRICTED_TEXT' });
      return;
    }

    this.router.navigate(['/register']);
  }
}
