import { HttpClient } from '@angular/common/http';
import { Component } from '@angular/core';
import { FormGroup, FormBuilder, Validators } from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';
import { SwalService } from '@app/shared/services/SwalService';
import { FormControlObject } from '@app/shared/types/FormFieldObject';
import { CustomValidators } from '@app/shared/validators/CustomValidators';

@Component({
  selector: 'app-change-password',
  templateUrl: './change-password-component.html',
  styleUrls: ['./change-password-component.scss'],
})
export class ChangePasswordComponent {
  public changePasswordForm: FormGroup;

  public fields: FormControlObject[] = [
    { label: 'New password', formControlName: 'password' },
    { label: 'Confirm new password', formControlName: 'confirmPassword' },
  ];

  public design = {
    header: 'no-bold',
    field: 'basic'
  };

  constructor(
    private readonly formBuilder: FormBuilder,
    private readonly router: Router,
    private readonly httpClient: HttpClient,
    private readonly activatedRoute: ActivatedRoute,
    private readonly swalService: SwalService
  ) {
    const password = this.formBuilder.control('', [
      Validators.required,
      Validators.minLength(8),
      Validators.maxLength(14),
      Validators.pattern(/^(?=.*[A-Z])(?=.*[0-9])(?=.{8,})/)
    ]);

    this.changePasswordForm = this.formBuilder.group({
      password,
      confirmPassword: ['', [Validators.required, CustomValidators.match(password)]],
    });
  }

  public changePassword(): void {
    const code = this.activatedRoute.snapshot.paramMap.get('code');

    if (this.changePasswordForm.invalid || !code || code === '') {
      return;
    }

    const formData: any = {
      password: this.changePasswordForm.get('password').value,
      code,
    };

    this.httpClient.disableAuthToken().post('/user/password/reset', formData).subscribe(data => {
      this.swalService.showSuccess({ title: 'Success', text: 'Password has been changed' });
      this.router.navigateByUrl('/confirm-change-password');
    }, error => {
      this.swalService.showException(error);
    });
  }
}
