import { AfterViewInit, Component, OnDestroy, OnInit } from '@angular/core';
import { MediaObserver } from '@angular/flex-layout';
import { Router } from '@angular/router';
import { AuthenticationService } from '@app/core/authentication/AuthenticationService';
import { I18nService } from '@app/core/I18nService';
import { UserService } from '@app/shared/services/UserService';
import { AppReadyEvent } from '@app/shared/AppReadyEvent';
import { User } from '@app/shared/types/User';
import { HttpClient } from '@angular/common/http';
import { SwalService } from '@app/shared/services/SwalService';
import { SystemEventBus } from '@app/shared/event-bus/system.event.bus';

export interface NavArray {
  title: string;
  icon: string;
  route: string;
  childRoutes?: NavArray[];
  alert: boolean;
}

@Component({
  selector: 'app-main-layout',
  templateUrl: './main-layout-component.html',
  styleUrls: ['./main-layout-component.scss'],
})
export class MainLayoutComponent implements OnInit, AfterViewInit, OnDestroy {
  private eventListener: (...args: any[]) => void;

  public navArray: NavArray[] = [
    { title: 'Wallets', icon: 'accounts', route: 'wallets', alert: false },
    { title: 'Vault', icon: 'vault', route: 'vault', alert: false },
    { title: 'Transactions', icon: 'transactions', route: 'transactions', alert: true },
    { title: 'Market', icon: 'pricing', route: 'market', alert: false },
    { title: 'Security', icon: 'settings', route: 'security', alert: false },
  ];

  public infoMenuArray = [
    { title: 'Desktop key', route: 'desktop' },
    { title: 'Mobile key', route: 'mobile' },
    { title: 'Help center', route: 'help' },
    { title: 'Privacy policy', route: 'privacy-policy' },
    { title: 'Terms and conditions', route: 'terms-conditions' },
  ];

  public currentDate = new Date();
  public profileImage: string;
  public sideNavActive = false;
  public activeRoute: any;
  public hideBalanceValue = true;

  public firstName = '';
  public lastName = '';

  constructor(
    private readonly router: Router,
    private readonly media: MediaObserver,
    private readonly authenticationService: AuthenticationService,
    private readonly i18nService: I18nService,
    private readonly userService: UserService,
    private readonly httpClient: HttpClient,
    private readonly swalService: SwalService,
    private readonly systemEventBus: SystemEventBus,
    private readonly appReadyEvent: AppReadyEvent
  ) {
  }

  ngOnInit(): void {
    this.userService.getMe().subscribe((data: any) => {
        const user = data.user as User;
        this.firstName = user.profile.firstName || '';
        this.lastName = user.profile.lastName || '';
      },
      (err) => {
        console.error(err);
      }
    );

    this.eventListener = (evt, ...args) => this.onSystemSettingsChanged(evt, ...args);
    this.systemEventBus.addListener('system_settings_changed', this.eventListener);

    this.refresh();
  }

  ngAfterViewInit(): void {
    setTimeout(() => {
      this.appReadyEvent.trigger();
    }, 500);
  }

  ngOnDestroy(): void {
    this.systemEventBus.removeListener('system_settings_changed', this.eventListener);
  }

  refresh(): void {
    this.receiveHideBalance();
  }

  public receiveHideBalance(): void {
    this.httpClient.get<boolean>('/user/settings/hide-balances').subscribe(
      (hideBalance) => {
        this.userService.reload();
        this.hideBalanceValue = hideBalance;
      },
      (err) => {
        this.swalService.showError({ title: 'Error', text: err.message });
      }
    );
  }

  toggleHideBalance(): void {
    this.hideBalanceValue = !this.hideBalanceValue;
    this.httpClient.put<boolean>('/user/settings/hide-balances', {on: this.hideBalanceValue}).subscribe(
      (hideBalance) => {
        this.hideBalanceValue = hideBalance;
        this.systemEventBus.triggerEvent('system_settings_changed', 'hide-balances', this.hideBalanceValue);
      },
      (err) => {
        this.receiveHideBalance();
        this.swalService.showError({ title: 'Error', text: err.message });
      }
    );
  }

  goToProfile(): void {
    this.router.navigateByUrl('/profile');
  }

  public getRouterLink(route: string): boolean {
    if (this.router.url.includes(route)) {
      return true;
    }
    else {
      return false;
    }
  }

  get isMobile(): boolean {
    return this.media.isActive('xs') || this.media.isActive('sm');
  }

  setLanguage(language: string): void {
    this.i18nService.language = language;
  }

  logout(): void {
    this.authenticationService
      .logout()
      .subscribe(() => this.router.navigate(['/login'], { replaceUrl: true }));
  }

  get name(): string | null {
    // const name = this.userService.get('firstName') + ' ' + this.userService.get('lastName');
    const name = `${this.firstName} ${this.lastName}`;
    // TODO: Always true
    return name ? name : null;
  }

  get email(): string | null {
    return this.userService.get('email');
  }

  get avatar(): string {
    const avatar = null; // this.userService.get('avatar');
    return avatar ? avatar : '/assets/images/user-blank.jpg';
  }

  get hideBalance(): boolean {
    const hideValues = this.userService.getDefaultShowBalance();
    return hideValues ? hideValues : false;
  }

  private onSystemSettingsChanged(...args: any[]): void {
    console.log(`${this.name}`);
    this.refresh();
  }
}
