import { Subject, Subscription } from 'rxjs';
import { SocketService } from './SocketService';

export abstract class SocketSubServiceBase {
  protected eventSubject: Subject<{name: string, args?: any[]}>;
  protected socket: SocketIOClient.Socket;

  constructor(
    protected readonly namespace: string,
    protected readonly socketService: SocketService
  ) {
    this.eventSubject = new Subject<{name: string, args?: any[]}>();
    this.socketService.connectNsp(namespace, (socket) => {
      this.socket = socket;
      this.socketService.onConnected(this.onConnectWrapper.bind(this));
      this.socketService.onDisconnected(this.onDisconnectWrapper.bind(this));
      this.socketService.onClear(this.onClearWrapper.bind(this));
      this.onConnectWrapper();
      this.initListeners();
    });
  }

  private onConnectWrapper(): void {
    if (this.onConnect){
      this.onConnect();
    }
  }
  private onDisconnectWrapper(): void {
    if (this.onDisconnect){
      this.onDisconnect();
    }
  }
  private onClearWrapper(): void {
    if (this.onClear){
      this.onClear();
    }
  }
  protected onConnect?(): any;
  protected onDisconnect?(): any;

  protected onEvent(name: string, cb: (...args: any[]) => void): Subscription;
  protected onEvent(name: string, cb: (...args: any[]) => void, once: boolean): void;

  protected onEvent(name: string, cb: (...args: any[]) => void, once?: boolean): Subscription | void {
    const subs = this.eventSubject.subscribe(ob => {
      if (ob.name !== name){
        return;
      }
      cb(...ob.args);
      if (once){
        subs.unsubscribe();
      }
    });
    if (!once){
      return subs;
    }
  }

  protected abstract initListeners(): void;
  protected abstract onClear(): void;
}
