import { Injectable } from '@angular/core';
import { environment } from '@env/environment';
import { Subject, Subscription } from 'rxjs';
import * as io from 'socket.io-client';

@Injectable()
export class SocketService {
  public eventSubject: Subject<{name: string, args?: any[]}>;
  private socket: SocketIOClient.Socket;
  private namespaces: SocketIOClient.Socket[] = [];
  private connecting = false;

  constructor() {
    this.eventSubject = new Subject<{name: string, args?: any[]}>();
  }

  public connect(token: string): void {
    if (this.connecting || (this.socket && this.socket.connected)){
      return;
    }
    this.connecting = true;
    if (!this.socket){
      this.socket = io.connect(environment.socketHost, {transports: ['websocket'], query: {token}});
      this.socket.on('error', this.onSocketError.bind(this));
      this.socket.on('connect', this.onConnect.bind(this));
      // this.socket.on('reconnect', this.onConnect.bind(this));
      this.socket.on('disconnect', this.onDisconnect.bind(this));
    } else {
      this.socket.io.opts.query = {token};
      this.socket.connect();
    }
  }

  public disconnect(): void {
    if (this.socket){
      this.socket.disconnect();
    }
    this.eventSubject.next({name: 'clear'});
    if (this.connecting){
      this.connecting = false;
    }
  }

  public on(name: string, cb: (...args: any[]) => void): Subscription {
    return this.eventSubject.subscribe((ob) => {
      if (!ob.name || ob.name !== name) {
        return;
      }
      cb(...ob.args);
    });
  }

  public get connected(): boolean {
    return this.socket && this.socket.connected;
  }

  public connectNsp(nsp: string, onConnect: (socket: SocketIOClient.Socket) => void): void{
    if (this.connected){
      const sock = io.connect(environment.socketHost + nsp, {transports: ['websocket']});
      this.namespaces.push(sock);
      onConnect(sock);
    } else {
      const once = this.onConnected(() => {
        const sock = io.connect(environment.socketHost + nsp, {transports: ['websocket']});
        this.namespaces.push(sock);
        onConnect(sock);
        once.unsubscribe();
      });
    }
  }

  public onError(cb: (error: any) => void): Subscription{
    return this.eventSubject.subscribe((ob) => {
      if (ob.name === 'error'){
        cb(ob.args[0]);
      }
    });
  }

  public onConnected(cb: () => void): Subscription{
    return this.eventSubject.subscribe((ob) => {
      if (ob.name === 'connected'){
        cb();
      }
    });
  }

  public onDisconnected(cb: () => void): Subscription{
  return this.eventSubject.subscribe((ob) => {
      if (ob.name === 'disconnected'){
        cb();
      }
    });
  }

  public onClear(cb: () => void): Subscription{
    return this.eventSubject.subscribe((ob) => {
      if (ob.name === 'clear'){
        cb();
      }
    });
  }

  private onSocketError(error: any): void {
    this.eventSubject.next({name: 'error', args: [error]});
  }

  private onConnect(): void {
    // conenct to all namespaces
    for (const nsp of this.namespaces){
      if (!nsp.connected){
        nsp.connect();
      }
    }
    this.connecting = false;
    this.eventSubject.next({name: 'connected'});
  }

  private onDisconnect(): void {
    // disconnect all namespaces
    for (const nsp of this.namespaces) {
      if (nsp.connected){
        nsp.disconnect();
      }
    }
    this.connecting = false;
    this.eventSubject.next({name: 'disconnected'});
  }
}
