import { Component, OnInit, OnDestroy, Renderer2 } from '@angular/core';
import { Router, NavigationEnd, ActivatedRoute } from '@angular/router';
import { Title } from '@angular/platform-browser';
import { TranslateService } from '@ngx-translate/core';
import { merge } from 'rxjs';
import { filter, map, mergeMap } from 'rxjs/operators';
import * as countries from 'i18n-iso-countries';
import { environment } from '@env/environment';
import { Logger } from './core/LoggerService';
import { I18nService } from './core/I18nService';
import { untilDestroyed } from './core/UntilDestroyed';
import { Angulartics2GoogleAnalytics } from 'angulartics2/ga';
import { IconService } from './shared/services/IconService';

@Component({
    selector: 'app-root',
    templateUrl: './app-component.html',
    styleUrls: ['./app-component.scss']
})
export class AppComponent implements OnInit, OnDestroy {
    private readonly logger = new Logger(AppComponent.name);

    constructor(
        private router: Router,
        private activatedRoute: ActivatedRoute,
        private titleService: Title,
        private translateService: TranslateService,
        private i18nService: I18nService,
        private angulartics2GoogleAnalytics: Angulartics2GoogleAnalytics,
        private renderer: Renderer2,
        private iconService: IconService
    ) {
        if (environment.production) {
            this.angulartics2GoogleAnalytics.startTracking();
        }
    }

    ngOnInit(): void {
        // Setup logger
        if (environment.production) {
            Logger.enableProductionMode();
        }

        if (environment.sumSubScript !== undefined) {
            this.addJsToElement(environment.sumSubScript);
        }

        // Setup translations
        this.i18nService.init(environment.defaultLanguage, environment.supportedLanguages);
        const onNavigationEnd = this.router.events.pipe(filter(event => event instanceof NavigationEnd));

        countries.registerLocale(require('i18n-iso-countries/langs/en.json'));

        // Change page title on navigation or language change, based on route data
        merge(this.translateService.onLangChange, onNavigationEnd)
            .pipe(
                map(() => {
                    let route = this.activatedRoute;
                    while (route.firstChild) {
                        route = route.firstChild;
                    }
                    return route;
                }),
                filter(route => route.outlet === 'primary'),
                mergeMap(route => route.data),
                untilDestroyed(this)
            )
            .subscribe(event => {
                // TODO: What?
                // tslint:disable-next-line: no-string-literal
                const title = event['title'];
                if (title) {
                    // this.titleService.setTitle(this.translateService.instant(title));
                    this.titleService.setTitle('NXACustody');
                }
            });

        // Load custom icons
        this.iconService.loadCustomIcons();

    }

    ngOnDestroy(): void {
        this.i18nService.destroy();
    }

    addJsToElement(src: string): HTMLScriptElement {
        const script = document.createElement('script');
        script.type = 'text/javascript';
        script.src = src;
        this.renderer.appendChild(document.body, script);
        return script;
    }
}
