import { DvitaRewardRepository } from "../repository/dvita.reward.repository";
import { HttpException, Injectable, Logger } from "@nestjs/common";
import { WelcomeRewardRequest } from "./dto/welcome.reward.request";
import { WelcomeRewardResult } from "./dto/welcome.reward.result";
import { wallet} from "@cityofzion/neon-core";
import { BlockchainService } from "@/blockchain/blockchain.service";
import { Campaign } from "./dto/welcome.reward.enum";

@Injectable()
export class WelcomeService {
    // 1 DVT
    private static readonly dvtRewardAmount = 1;
    // 10 DVG in decimals scale
    private static readonly dvgRewardAmount = 10_0000_0000;
    private static readonly campaignId = Campaign.Welcome;
    private readonly logger = new Logger(WelcomeService.name);
    private static readonly hotWatllet = process.env.BLOCKCHAIN_SYS_HOT_PK!;
    private static readonly dvitaHash = process.env.DVITA_TOKEN_HASH!;
    private static readonly dvgHash = process.env.DVP_TOKEN_HASH!;

    constructor(
      private readonly welcomeRewardRepository: DvitaRewardRepository,
      private readonly blockchainService: BlockchainService
    ) {
    }

    async reward(dto: WelcomeRewardRequest): Promise<WelcomeRewardResult> {
        const existingRewards = await this.welcomeRewardRepository.find({
          where: 
          {
            userId: dto.userId, 
            socialNetwork: dto.socialNetwork, 
            campaignId: WelcomeService.campaignId
          }
        });
        
        if(existingRewards.length > 0){
          this.logger.error(`Reward already received: Social network ${dto.socialNetwork} for user id ${dto.userId} and campaign id ${WelcomeService.campaignId}.`);
          throw new HttpException("Reward already received.", 409);
        }

        const fromAccount = new wallet.Account(WelcomeService.hotWatllet);
        const dvitaTx = await this.blockchainService.sendTx(fromAccount, dto.address, WelcomeService.dvitaHash, WelcomeService.dvtRewardAmount);
        this.logger.log(dvitaTx);

        const dvgTx = await this.blockchainService.sendTx(fromAccount, dto.address, WelcomeService.dvgHash, WelcomeService.dvgRewardAmount);
        this.logger.log(dvgTx);
        
        const welcomeRewardEntity = this.welcomeRewardRepository.create();
        welcomeRewardEntity.userId = dto.userId;
        welcomeRewardEntity.userSocialHandle = dto.userSocialHandle;
        welcomeRewardEntity.address = dto.address;
        welcomeRewardEntity.dvitaAmount = WelcomeService.dvtRewardAmount.toString();
        welcomeRewardEntity.dvitaTxHash = dvitaTx;
        welcomeRewardEntity.dvgAmount = WelcomeService.dvgRewardAmount.toString();
        welcomeRewardEntity.dvgTxHash = dvgTx;
        welcomeRewardEntity.accessToken = dto.accessToken;
        welcomeRewardEntity.socialNetwork = dto.socialNetwork;
        welcomeRewardEntity.campaignId = WelcomeService.campaignId;

        const savedEntity = await this.welcomeRewardRepository.save(welcomeRewardEntity);

        const result = new WelcomeRewardResult(
            savedEntity.dvitaTxHash, 
            savedEntity.dvgTxHash, 
            savedEntity.dvitaAmount, 
            savedEntity.dvgAmount,
            savedEntity.address, 
            savedEntity.userId
          );

        return result;
    }
}