import { NestFactory } from '@nestjs/core';
import { Logger, INestApplication } from '@nestjs/common';
import { AppModule } from './app/app.module';
import { ConfigService } from '@nestjs/config';
import * as bodyParser from 'body-parser';
import { SwaggerModule, DocumentBuilder } from '@nestjs/swagger';
import { NestExpressApplication } from '@nestjs/platform-express';

async function bootstrapSwagger(app: INestApplication) {
  const options = new DocumentBuilder()
      .setTitle('DVITA Social Rewards API')
      .setDescription('DVITA Social Rewards API - The Social Rewards microservice is a digital loyalty platform for social network users to earn dvg & dvt for their activities.')
      .setVersion('1.0')
      .build();
  const document = SwaggerModule.createDocument(app, options);
  SwaggerModule.setup('api-doc', app, document);
}

async function bootstrap() {
  // Application
  const app = await NestFactory.create<NestExpressApplication>(AppModule, {
      logger: ['log', 'error', 'warn', 'debug', 'verbose']
  });

  // Global Prefix
  app.setGlobalPrefix('/api');
  
  // CORS
  app.enableCors();

  // Swagger
  await bootstrapSwagger(app);

  // Body Parser Limits
  app.use(bodyParser.json({ limit: '50mb' }));
  app.use(bodyParser.urlencoded({ limit: '50mb', extended: true }));
  app.use(bodyParser.raw({ inflate: true, limit: '50mb', type: 'multipart/form-data' }));

  // Logger
  const logger = new Logger(AppModule.name);

  // Config Service
  const configService = app.get<ConfigService>(ConfigService);

  // HTTP Service
  const httpPortString = configService.get('API_HTTP_PORT');
  await app.listen(httpPortString);

  const appUrl = await app.getUrl();
  logger.verbose(`Application is running at: ${appUrl}, http port: ${httpPortString}`);
}

bootstrap();