import { Injectable, Logger } from "@nestjs/common";
import { rpc, wallet, sc, tx, CONST, u } from "@cityofzion/neon-core";

@Injectable()
export class BlockchainService {
    private client: rpc.RPCClient | undefined;
    private readonly logger = new Logger(BlockchainService.name);
    private static readonly networkMagic = Number(process.env.NETWORK_MAGIC!);
    private static readonly networkUrl = process.env.NODE_URL!;

      connect(): void {
        this.logger.log(`Connecting to '${BlockchainService.networkUrl}.'`);
        this.client = new rpc.RPCClient(BlockchainService.networkUrl);
      }
  
      disconnect(): void {
        this.logger.log('Disconnecting from node.');
        this.client = undefined;
      }

      async sendTx(fromAccount: wallet.Account, toAddress: string, tokenHash: string, amount: number): Promise<string>{
        const script = sc.createScript({
            scriptHash: tokenHash,
            operation: "transfer",
            args: [
                sc.ContractParam.hash160(fromAccount.address),
                sc.ContractParam.hash160(toAddress),
                sc.ContractParam.integer(amount),
                sc.ContractParam.any(),
              ],
            });
        
            const currentHeight = await this.client!.getBlockCount();
            const transaction = new tx.Transaction({
              signers: [
                {
                  account: fromAccount.scriptHash,
                  scopes: tx.WitnessScope.CalledByEntry
                },
              ],
              validUntilBlock: currentHeight + 10,
              script
            });
  
        await this.calculateNetworkFee(transaction);
        await this.calculateSystemFee(transaction, fromAccount);
        transaction.sign(fromAccount, BlockchainService.networkMagic);
        const txResult = await this.client!.sendRawTransaction(transaction);
        return txResult;
      }
  
      async calculateNetworkFee(transaction: tx.Transaction): Promise<void> {
        this.logger.log("Calculating network fee");
  
        const feePerByteResponse = await this.client!.invokeFunction(
          CONST.NATIVE_CONTRACT_HASH.PolicyContract,
          "getFeePerByte"
        );
          
        if (feePerByteResponse.state !== "HALT") {
          throw new Error(`Unable to retrieve data to calculate network fee. State: ${feePerByteResponse.state}, Exception: ${feePerByteResponse.exception}`);
        }
  
        const feePerByte = feePerByteResponse.stack[0].value as number;
        const feePerByteBigInt = u.BigInteger.fromNumber(feePerByte);
  
        // Account for witness size
        const transactionByteSize = transaction.serialize().length / 2 + 109;
  
        // Hardcoded. Running a witness is always the same cost for the basic account.
        const witnessProcessingFee = u.BigInteger.fromNumber(1000390);
        const networkFeeEstimate = feePerByteBigInt
          .mul(transactionByteSize)
          .add(witnessProcessingFee);
  
        transaction.networkFee = networkFeeEstimate;
        this.logger.log(networkFeeEstimate);
      }
  
      async calculateSystemFee(transaction: tx.Transaction, fromAccount: wallet.Account): Promise<void> {
        this.logger.log("Calculating system fee");
        const signers:any = 
        [ 
          {
            account: fromAccount.scriptHash,
            scopes: tx.WitnessScope.CalledByEntry
          } 
        ];
  
        const invokeFunctionResponse = await this.client!.invokeScript(
          u.HexString.fromHex(transaction.script), 
          signers
        );
        
        this.logger.log(invokeFunctionResponse);
        
        if (invokeFunctionResponse.state !== "HALT") {
          throw new Error(`Transfer script errored out! State: ${invokeFunctionResponse.state}, Exception: ${invokeFunctionResponse.exception}`);
        }
  
        const requiredSystemFee = u.BigInteger.fromNumber(
          invokeFunctionResponse.gasconsumed
        );
        transaction.systemFee = u.BigInteger.fromNumber(Number(requiredSystemFee));
        this.logger.log(requiredSystemFee);
      }
}