import { BadRequestException, Injectable } from '@nestjs/common';
import { CreateTalismanBidDto } from './dto/create.talisman.bid.dto';
import { TalismanBidDto } from './dto/talisman.bid.dto';
import { TalismanBidResultDto } from './dto/talisman.bid.result.dto';
import { TalismanBidRepository } from './talisman.bid.repository';

@Injectable()
export class TalismanService {
    constructor(
        private readonly talismanBidRepository: TalismanBidRepository
    ) {
    }

    async getAllBids(): Promise<TalismanBidDto[]> {
        const entities = await this.talismanBidRepository.find({
            order: {
                price: 'DESC'
            }
        });

        if (!entities || entities.length <= 0) {
            return [];
        }

        const dtos = entities.map(TalismanBidDto.fromEntity);

        return dtos;
    }

    async getHighestBid(): Promise<TalismanBidDto> {
        const entities = await this.talismanBidRepository.find({
            order: {
                price: 'DESC'
            },
            take: 1
        });

        if (!entities || entities.length <= 0) {
            // 'USD' is default & hardcoded currency ATM.
            // See https://andit.atlassian.net/browse/NFT-26
            const emptyDto = new TalismanBidDto('USD', 0, undefined);
            return emptyDto;
        }

        const dto = TalismanBidDto.fromEntity(entities[0]);

        return dto;
    }

    async createTalismanBid(dto: CreateTalismanBidDto): Promise<TalismanBidResultDto> {
        const highestBid = await this.getHighestBid();
        const isHighest = highestBid.price < dto.price;

        if (!isHighest) {
            throw new BadRequestException('Bid price is less or equal than current highest bid');
        }

        const created = this.talismanBidRepository.create(dto);
        const saved = await this.talismanBidRepository.save(created);
        const savedDto = TalismanBidResultDto.fromEntity(saved, isHighest);
        return savedDto;
    }
}
