import { ApiProperty, ApiPropertyOptional } from "@nestjs/swagger";
import { Transform } from "class-transformer";
import { IsBoolean, IsDate, IsEmail, IsNotEmpty, IsNumber, IsString, MaxLength, MinLength } from "class-validator";
import { TalismanBidEntity } from "../../entity/talisman.bid.entity";

export class TalismanBidResultDto {
    @ApiProperty()
    @IsEmail()
    @IsNotEmpty()
    @MinLength(4)
    @MaxLength(255)
    email: string;

    @ApiProperty()
    @IsString()
    @IsNotEmpty()
    @MinLength(2)
    @MaxLength(16)
    currencyCode: string;

    @ApiProperty()
    @IsNumber({ allowInfinity: false, allowNaN: false })
    @Transform(params => parseFloat(params.value))
    price: number;

    @ApiProperty()
    @IsBoolean()
    highest: boolean;

    @ApiPropertyOptional()
    lotId?: number;

    @ApiProperty()
    @IsDate()
    createdDate: Date;

    constructor(
        email: string,
        currencyCode: string,
        price: number,
        highest: boolean,
        createdDate: Date,
        lotId?: number
    ) {
        this.email = email;
        this.currencyCode = currencyCode;
        this.price = price;
        this.highest = highest;
        this.createdDate = createdDate;
        this.lotId = lotId;
    }

    public static fromEntity(entity: TalismanBidEntity, highest: boolean): TalismanBidResultDto {
        return new TalismanBidResultDto(
            entity.email,
            entity.currencyCode,
            entity.price,
            highest,
            entity.createdDate,
            entity.lotId
        );
    }
}
