import { ThrowStmt } from '@angular/compiler';
import { Component, ElementRef, OnInit, ViewChild } from '@angular/core';
import { FormArray, FormBuilder, FormControl, FormGroup, Validators } from '@angular/forms';
import { SnackBarService } from '@app/shared/services/snack-bar.service';
import { wallet } from '@cityofzion/neon-core';

@Component({
  selector: 'app-multisig-wallet',
  templateUrl: './multisig-wallet.component.html',
  styleUrls: ['./multisig-wallet.component.scss'],
})
export class MultisigWalletComponent implements OnInit {
  // publicKeys = ['', '', ''];
  verificationScript = '';

  // publicKeysForm = this.formBuilder.group({
  //   publicKey: new FormControl('', [Validators.required]),
  // });

  multiSignatureForm = this.formBuilder.group({
    // publicKeys: new FormControl('', [Validators.required]),
    keys: this.formBuilder.array([])
  })

  @ViewChild('result') result!: ElementRef;

  constructor(
    private readonly snackBarService: SnackBarService,
    private formBuilder: FormBuilder
  ) { }

  ngOnInit() {
    this.keys.push(new FormControl('', Validators.required));
  }

  get keys() {
    return this.multiSignatureForm.controls["keys"] as FormArray;
  }

  addKey() {
    if (this.keys.length < 10) {
      this.keys.push(new FormControl('', Validators.required));
    }
  }

  removeKey(index: number): void {
    if (this.keys.length > 1) {
      this.keys.removeAt(index);
    }
  }

  resetValidity(event: Event): void {
    const el = event.target as HTMLInputElement;
    el.setCustomValidity('');
    this.verificationScript = '';
  }

  submit(event: Event): void {
    const form = event.target as HTMLFormElement;
    const formData = new FormData(form);
    const elements = form.elements as HTMLFormControlsCollection & Record<string, HTMLInputElement>;

    // validate public keys
    const publicKeys: string[] = []
    formData.forEach((rawValue, keys) => {
      const value = (rawValue as string).trim();
      if (keys.startsWith('publicKey') && value) {
        if (publicKeys.includes(value)) {
          elements[keys].setCustomValidity(`No duplicates please`);
          elements[keys].reportValidity();
        } else if (!wallet.isPublicKey(value)) {
          elements[keys].setCustomValidity('This public key is invalid');
          elements[keys].reportValidity();
        } else {
          publicKeys.push(value);
        }
      }
    });

    // validate threshold
    const threshold = Number(formData.get('threshold'));
    if (form.checkValidity()) {
      if (threshold < 1 || threshold > this.keys.length) {
        elements.threshold.setCustomValidity('Number must be the amount of public keys');
        elements.threshold.reportValidity();
      }
    }

    if (form.checkValidity()) {
      // Sort the keys in the same way as neo-cli is sorting them, so that the generated address matches neo-cli
      // https://github.com/neo-project/neo/blob/master/src/neo/SmartContract/Contract.cs#L108
      const keys: string[] = this.multiSignatureForm?.get('keys')?.value;
      const sortedKeys = keys
        .map(k => k ? wallet.getPublicKeyUnencoded(k) : k)
        .filter(k => !!k)
        .sort()
        .map(k => k ? wallet.getPublicKeyEncoded(k) : k);

      if (sortedKeys.length >= threshold) {
        this.verificationScript = wallet.constructMultiSigVerificationScript(threshold, sortedKeys);
        this.result.nativeElement.scrollIntoView();
      }
    }
  }

  getAddress(): string {
    return wallet.getAddressFromScriptHash(wallet.getScriptHashFromVerificationScript(this.verificationScript));
  }

  copy(text: string): void {
    navigator.clipboard
      .writeText(String(text))
      .then(() => this.snackBarService.show('Copied'))
      .catch(() => this.snackBarService.show('Could not copy, please do it manually'));
  }
}
