using Akka.Actor;
using Neo;
using Neo.Network.P2P.Payloads;
using Neo.Persistence;
using Neo.SmartContract;
using Neo.SmartContract.Native;
using Neo.Wallets;

namespace Dvita.SmartContract.Wrapper
{
    public class TransactionWrapper
    {
        #nullable disable
        public static DataCache Snapshot { get; set; }
        public static Transaction Transaction { get; set; }
        public static Wallet CurrentWallet { get; set; }
        public static NeoSystem System { get; set; }
        #nullable restore

        public TransactionWrapper(Transaction tx, DataCache snapshot, Wallet wallet, NeoSystem system) 
        {
            Snapshot = snapshot;
            Transaction = tx;
            CurrentWallet = wallet;
            System = system;
        }

        public TransactionState Wait(double deadlineSeconds = 60)
        {
            var deadline = DateTime.UtcNow.AddSeconds(deadlineSeconds);
            Transaction? tx = null;
            
            var faultState = new TransactionState() { State = Neo.VM.VMState.FAULT, Transaction = null };
            TransactionState txState = faultState;
            
            SendTx();

            while (tx == null)
            {
                if (deadline < DateTime.UtcNow)
                {
                    throw new TimeoutException();
                }

                txState = NativeContract.Ledger.GetTransactionState(Snapshot, Transaction.Hash);
                tx = txState?.Transaction;

                if (tx == null)
                {
                    Task.Delay(((int)(ProtocolSettings.Default.MillisecondsPerBlock / 2)));
                }
            }

            return txState ?? faultState;
        }

        public async Task<TransactionState> WaitAsync(CancellationToken cancellationToken)
        {
            Transaction? tx = null;

            var faultState = new TransactionState() { State = Neo.VM.VMState.FAULT, Transaction = null };
            TransactionState txState = faultState;
            
            await Task.Run(() =>
            {
                SendTx();

                while (tx == null)
                {
                    txState = NativeContract.Ledger.GetTransactionState(Snapshot, Transaction.Hash);
                    tx = txState?.Transaction;

                    if (tx == null)
                    {
                        Task.Delay(((int)(ProtocolSettings.Default.MillisecondsPerBlock / 2)));
                    }
                }
            }, cancellationToken);

            return txState;
        }

        private void SendTx()
        {
            var context = new ContractParametersContext(Snapshot, Transaction, System.Settings.Network);
            CurrentWallet.Sign(context);
            
            if (!context.Completed)
            {
                throw new InvalidOperationException("Witnesses are not ready to be added.");
            }

            Transaction.Witnesses = context.GetWitnesses();
            System.Blockchain.Tell(Transaction);
        }
    }
}
