using Neo.IO.Json;
using Neo.Network.P2P.Payloads;
using Neo.Persistence;
using Neo.SmartContract.Manifest;
using Neo.SmartContract.Native;
using Neo.VM;
using Neo.Wallets;
using System.Reflection;
using Neo;
using Neo.SmartContract;
using Microsoft.Extensions.Logging;
using Neo.VM.Types;

namespace Dvita.SmartContract.Wrapper
{
    public class SmartContractWrapper<T>
    {
        #nullable disable
        public UInt160 ScriptHash { get; set; }
        public ContractState ContractState { get; set; }
        public string SignerPrivateKey { get; set; }
        public string SignerPublicKey { get; set; }
        public string OwnerAddress { get; set; }
        public UInt160 SignerScriptHash { get; set; }
        public KeyPair SignerKeyPair { get; set; }
        public DataCache Snapshot { get; set; }
        public ProtocolSettings Settings { get; set; }
        public Wallet CurrentWallet { get; set; }
        public WeakReference Logger { get; set; }
        public Signer[] Signers { get; set; }
        #nullable restore

        public SmartContractWrapper(DataCache snapshot, Wallet wallet, ProtocolSettings? settings = null)
        {
            Snapshot = snapshot;
            Settings = settings ?? ProtocolSettings.Default;
            CurrentWallet = wallet;
        }

        public SmartContractWrapper<T> WithScriptHash(string hash)
        {
            UInt160.TryParse(hash, out var contractHashUint);
            ScriptHash = contractHashUint;
            return this;
        }

        public SmartContractWrapper<T> WithName(string name)
        {
            var contract = NativeContract.Contracts.SingleOrDefault(c => c.Name == name);
            if (contract != null)
            {
                ScriptHash = contract.Hash;
            }

            return this;
        }

        public SmartContractWrapper<T> WithSignerPrivateKey(string privKey)
        {
            SignerKeyPair = new KeyPair(Wallet.GetPrivateKeyFromWIF(privKey));
            var contract = new Contract
            {
                Script = Contract.CreateSignatureRedeemScript(SignerKeyPair.PublicKey),
                ParameterList = new[] { ContractParameterType.Signature }
            };
            SignerScriptHash = contract.ScriptHash;
            SignerPrivateKey = privKey;
            CurrentWallet.Import(SignerPrivateKey);
            return this;
        }

        public SmartContractWrapper<T> WithSignerPublicKey(string pubKey)
        {
            SignerPublicKey = pubKey;
            return this;
        }

        public SmartContractWrapper<T> WithSignerScriptHash(UInt160 scriptHash)
        {
            SignerScriptHash = scriptHash;
            return this;
        }

        public SmartContractWrapper<T> WithOwnerAddress(string pubKey)
        {
            OwnerAddress = pubKey;
            return this;
        }

        public SmartContractWrapper<T> WithSigners(Signer[] signers)
        {
            Signers = signers;
            return this;
        }

        public SmartContractWrapper<T> WithLogger(ILogger<T> logger)
        {
            Logger = new WeakReference(logger);
            return this;
        }

        private ILogger<T>? GetLogger()
        {
            if (Logger != null && Logger.Target != null && Logger.IsAlive)
            {
                return Logger.Target as ILogger<T>;
            }

            return null;
        }

        public JObject GetABI()
        {
            return ContractState.Manifest.Abi.ToJson();
        }

        public ContractManifest GetManifest()
        {
            return ContractState.Manifest;
        }

        public ContractMethodDescriptor[] GetMethods()
        {
            return ContractState.Manifest.Abi.Methods;
        }

        public string[] GetSupportedStandards()
        {
            return ContractState.Manifest.SupportedStandards;
        }

        public string Name()
        {
            return ContractState.Manifest.Name;
        }

        public StackItem InvokeSafeMethod(string methodName, object[]? parameters = null)
        {
            GetLogger()?.LogInformation($"Invoking safe method {methodName}");
            var script = CreateScript(methodName, parameters ??  new object[]{});
            using var engine = ApplicationEngine.Run(script, Snapshot, null, null, Settings);
            return engine.ResultStack.Peek();
        }

        public Transaction InvokeMutatingMethod(string methodName, object[]? parameters = null)
        {
            GetLogger()?.LogInformation($"Invoking mutating method {methodName}");
            var script = CreateScript(methodName, parameters ??  new object[]{});
            var sender = CurrentWallet.GetAccounts()
                .SingleOrDefault(p => p.ScriptHash == SignerScriptHash) ?? CurrentWallet.GetAccounts().First();

            var signers = (Signers == null || Signers.Count() < 1) ? 
                new[] { new Signer() { Account = sender.ScriptHash, Scopes = WitnessScope.CalledByEntry } } : 
                Signers;

            return CurrentWallet.MakeTransaction(Snapshot, script, sender.ScriptHash, signers);
        }

        public byte[] CreateScript(string methodName, object[] parameters)
        {
            byte[] script;
            using (ScriptBuilder scriptBuilder = new ScriptBuilder())
            {
                scriptBuilder.EmitDynamicCall(ContractState.Hash, methodName, parameters);
                script = scriptBuilder.ToArray();
            }
            return script;
        }

        public dynamic Build()
        {
            if (ScriptHash == null)
            {
                throw new ArgumentNullException(nameof(ScriptHash));
            }

            ContractState = NativeContract.ContractManagement.GetContract(Snapshot, ScriptHash);

            System.Console.WriteLine(Assembly.GetCallingAssembly().GetName().Name!);
            System.Console.WriteLine(Assembly.GetExecutingAssembly().GetName().Name!);

            Type iLoadType = (from t in Assembly.Load(Assembly.GetExecutingAssembly().GetName().Name!).GetExportedTypes()
                              from t2 in Assembly.Load(Assembly.GetCallingAssembly().GetName().Name!).GetExportedTypes()
                              where !t.IsInterface && !t.IsAbstract && !t2.IsInterface && !t2.IsAbstract
                              where typeof(T).IsAssignableFrom(t) || typeof(T).IsAssignableFrom(t2)
                              select t).FirstOrDefault()!;

            return Activator.CreateInstance(iLoadType, Snapshot, CurrentWallet, Settings, this)!;
        }
    }
}


