using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;

namespace Dvita.SmartContract.Generator.Wrapper
{
    [Generator]
    public class SmartContractGenerator : ISourceGenerator
    {
        public void Initialize(GeneratorInitializationContext context)
        {
            context.RegisterForSyntaxNotifications(() => new AttributeSyntaxReceiver<GenerateSmartContractAttribute>());
        }

        public void Execute(GeneratorExecutionContext context)
        {
            if (context.SyntaxReceiver is not AttributeSyntaxReceiver<GenerateSmartContractAttribute> syntaxReceiver)
            {
                return;
            }
            var namespaceName = context.Compilation.AssemblyName;
            if (namespaceName == null)
                return;

            foreach (var interfaceSyntax in syntaxReceiver.Interfaces)
            {
                var interfaceName = interfaceSyntax.Identifier.ToString();
                var className = interfaceName.Substring(1);
                var methods = interfaceSyntax.Members.OfType<MethodDeclarationSyntax>().ToList();
                var modifiedMethods = new List<MethodDeclarationSyntax>();
                var ifacesInherited = interfaceSyntax?.BaseList?.Types;
                if (ifacesInherited != null)
                {
                    foreach (var ifaceInherited in ifacesInherited)
                    {
                        var root = ifaceInherited.SyntaxTree.GetRoot() as CompilationUnitSyntax;
                        if (root != null)
                        {
                            var nodeType = ifaceInherited.Type as IdentifierNameSyntax;
                            var inheritedMethods = root.DescendantNodes().OfType<MethodDeclarationSyntax>()
                                .Where(m => (m.Parent as InterfaceDeclarationSyntax)?.Identifier.ValueText == nodeType?.Identifier.ValueText);
                            methods.AddRange(inheritedMethods);
                        }
                    }
                }

                foreach (var method in methods)
                {
                    var newAttributes = new SyntaxList<AttributeListSyntax>();
                    var newStatement = SyntaxFactory.ParseStatement(GenerateMethodBody(method));
                    var newMethod = method.WithSemicolonToken(default(SyntaxToken))
                        .AddModifiers(SyntaxFactory.Token(SyntaxKind.PublicKeyword))
                        .WithAttributeLists(newAttributes)
                        .AddBodyStatements(newStatement);

                    modifiedMethods.Add(newMethod.NormalizeWhitespace());
                }

                var newNamespace = SyntaxFactory.NamespaceDeclaration(SyntaxFactory.ParseName(namespaceName))
                   .NormalizeWhitespace()
                   .AddUsings(CreateUsings());

                var classDeclaration = SyntaxFactory.ClassDeclaration(className)
                   .AddModifiers(SyntaxFactory.Token(SyntaxKind.PublicKeyword))
                   .AddBaseListTypes(
                       SyntaxFactory.SimpleBaseType(SyntaxFactory.ParseTypeName($"SmartContractWrapper<{interfaceName}>")),
                       SyntaxFactory.SimpleBaseType(SyntaxFactory.ParseTypeName(interfaceName)));


                var constructor = CreateConstructor(className, interfaceName);
                var members = new List<MemberDeclarationSyntax>(){ constructor }.Concat(modifiedMethods);
                classDeclaration = classDeclaration.WithMembers(new SyntaxList<MemberDeclarationSyntax>(members));
                newNamespace = newNamespace.WithMembers(new SyntaxList<MemberDeclarationSyntax>(classDeclaration));
                string source = newNamespace.NormalizeWhitespace().ToFullString();
                context.AddSource($"{className}.g.cs", source);
            }
        }

        private UsingDirectiveSyntax[] CreateUsings()
        {
            return new[]{
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Numerics")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.Persistence")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.Network.P2P.Payloads")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Dvita.SmartContract.Wrapper")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Dvita.SmartContract.Wrapper.Interface")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.SmartContract.Manifest")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.Wallets")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Reflection")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Linq")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.VM.Types")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.Cryptography.ECC")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.IO.Json")),
                SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Neo.SmartContract"))
        };
        }

        private ConstructorDeclarationSyntax CreateConstructor(string className, string interfaceName)
        {
            var foreachStatement = SyntaxFactory.ForEachVariableStatement(
                SyntaxFactory.ParseExpression("PropertyInfo property"),
                SyntaxFactory.ParseExpression($"typeof(SmartContractWrapper<{interfaceName}>).GetProperties().Where(p => p.CanWrite)"),
                SyntaxFactory.ParseStatement("property.SetValue(this, property.GetValue(wrapper, null), null);"));

            return SyntaxFactory.ConstructorDeclaration(className)
                    .AddParameterListParameters(
                        SyntaxFactory.Parameter(SyntaxFactory.Identifier("snapshot"))
                            .WithType(SyntaxFactory.ParseTypeName("DataCache")),
                        SyntaxFactory.Parameter(SyntaxFactory.Identifier("wallet"))
                            .WithType(SyntaxFactory.ParseTypeName("Wallet")),
                        SyntaxFactory.Parameter(SyntaxFactory.Identifier("settings"))
                            .WithType(SyntaxFactory.ParseTypeName("ProtocolSettings")),
                        SyntaxFactory.Parameter(SyntaxFactory.Identifier("wrapper"))
                            .WithType(SyntaxFactory.ParseTypeName($"SmartContractWrapper<{interfaceName}>"))
                        )
                    .WithBody(SyntaxFactory.Block(foreachStatement))
                    .AddModifiers(SyntaxFactory.Token(SyntaxKind.PublicKeyword))
                    .WithInitializer(SyntaxFactory.ConstructorInitializer(
                        SyntaxKind.BaseConstructorInitializer,
                            SyntaxFactory.ArgumentList(
                            SyntaxFactory.SeparatedList(new[] {
                                SyntaxFactory.Argument(
                                    SyntaxFactory.IdentifierName("snapshot")),
                                SyntaxFactory.Argument(
                                    SyntaxFactory.IdentifierName("wallet")),
                                SyntaxFactory.Argument(
                                    SyntaxFactory.IdentifierName("settings"))
                            }))));
        }

        private string GenerateMethodBody(MethodDeclarationSyntax m)
        {
            var parameterStr = string.Empty;
            var methodName=m.Identifier.ToString();
            var methodNameToLower = Helper.ToLowerCamelCase(methodName);
            var parameters = m.ParameterList.Parameters.Select(p => p.Identifier);
            var templateMethodName = m.AttributeLists.Any(al => al.Attributes.Any(a => a.Name.ToString().Equals(typeof(SafeMethod).Name))) ?
                "InvokeSafeMethod" : "InvokeMutatingMethod";
            if (parameters.Any())
            {
                parameterStr = $", new object[]{{ {string.Join(", ", parameters)} }}";
            }

            return @$"{Environment.NewLine}return {templateMethodName}(""{methodNameToLower}""{parameterStr});{Environment.NewLine}";
        }
    }
}
