using Neo.Network.P2P.Payloads;
using Neo.SmartContract;
using Neo.SmartContract.Native;
using Neo.Cryptography.ECC;
using Neo.Wallets.NEP6;
using Neo.Persistence;
using Dvita.SmartContract.Wrapper;
using Dvita.SmartContract.Wrapper.Interface;
using System;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Neo.UnitTests.SmartContract
{
    [TestClass]
    public class UT_SmartContractWrapper
    {
        const byte PrefixTransaction = 11;
        private NEP6Wallet testWallet => SetTestWallet();
        private string testSignerPrivKey => "KzoToapKJze1sFUsGvPC5Rv4t2Um9DnrwWUe6T59DebZF2rDqdH1";
        private string testSignerPubKey => "0xfa4b61e682edc9a8a7393e6879ea4902ce816f54";
        
        private NEP6Wallet SetTestWallet()
        {
            var wallet = new NEP6Wallet("./", ProtocolSettings.Default);
            wallet.Unlock("pass");
            wallet.CreateAccount();
            return wallet;
        }

        private void FakeBalance(UInt160 from, ref DataCache snapshot)
        {
            var key = NativeContract.GAS.CreateStorageKey(20, from);
            var entry = snapshot.GetAndChange(key, () => new StorageItem(new AccountState()));
            entry.GetInteroperable<AccountState>().Balance = 1000000 * NativeContract.GAS.Factor;
            snapshot.Commit();
        }

        private void FakeTx(Transaction tx, ref DataCache snapshot)
        {
            //Fake a transaction into snapshot
            var state = new TransactionState()
            {
                BlockIndex = 0,
                Transaction = tx
            };
            snapshot.Add(NativeContract.Ledger.CreateStorageKey(PrefixTransaction, tx.Hash), new StorageItem(state));
        }

        [TestMethod]
        public void TestWrapperForDvitaContractAndTransfer()
        {
            var snapshot = TestBlockchain.GetTestSnapshot();
            var serviceProvider = new ServiceCollection()
            .AddLogging()
            .BuildServiceProvider();
            var factory = serviceProvider.GetService<ILoggerFactory>()!;
            var logger = factory.CreateLogger<INep17ContractABI>();
            var contract = new SmartContractWrapper<INep17ContractABI>(snapshot, testWallet)
            .WithScriptHash("0xb34e1025391e953a918231df11478ec21b039e5f")
            .WithSignerPrivateKey(testSignerPrivKey)
            .WithSignerPublicKey(testSignerPubKey)
            .WithLogger(logger)
            .Build();

            Assert.AreEqual(contract.Name(), "DvitaToken");
            Assert.AreEqual(contract.Symbol(), "DVT");
            Assert.IsNotNull(contract.GetABI());
            Assert.AreEqual(contract.Logger.Target, logger);

            UInt160.TryParse(testSignerPubKey, out var from);
            UInt160.TryParse(testSignerPubKey, out var to);
            FakeBalance(from, ref snapshot);

            var tx = contract.Transfer(from, to, 1L, null) as Transaction;
            Assert.IsNotNull(tx?.Hash);
            FakeTx(tx, ref snapshot);

            var txWrapper = new TransactionWrapper(tx, snapshot, contract.CurrentWallet, TestBlockchain.TheNeoSystem);
            var txResult = txWrapper.Wait();
            Assert.IsNotNull(txResult.Transaction);
        }

        [TestMethod]
        public void TestWrapperForDvitaVote()
        {
            var snapshot = TestBlockchain.GetTestSnapshot();
            var serviceProvider = new ServiceCollection()
            .AddLogging()
            .BuildServiceProvider();
            var factory = serviceProvider.GetService<ILoggerFactory>()!;
            var logger = factory.CreateLogger<INep17ContractABI>();

            UInt160.TryParse(testSignerPubKey, out var from);
            UInt160.TryParse(testSignerPubKey, out var to);
            FakeBalance(from, ref snapshot);

            var contract = new SmartContractWrapper<IDVTContractABI>(snapshot, testWallet)
            .WithScriptHash("0xb34e1025391e953a918231df11478ec21b039e5f")
            .WithSignerPrivateKey(testSignerPrivKey)
            .WithSignerPublicKey(testSignerPubKey)
            .Build();

            Assert.AreEqual(contract.Name(), "DvitaToken");
            Assert.AreEqual(contract.Symbol(), "DVT");

            ECPoint voteForPubKey = ECPoint.Parse("0264b5ff308d12d9dcbcb929784278a666f988d8a9388e61653cf5c22cb9d05864", ECCurve.Secp256r1);

            var tx = contract.Vote(from, voteForPubKey) as Transaction;
            Assert.IsNotNull(tx?.Hash);
            FakeTx(tx, ref snapshot);

            var txWrapper = new TransactionWrapper(tx, snapshot, contract.CurrentWallet, TestBlockchain.TheNeoSystem);
            var txResult = txWrapper.Wait();
            Assert.IsNotNull(txResult.Transaction);
        }


        [TestMethod]
        public async Task TestWrapperForDVPContract()
        {
            var snapshot = TestBlockchain.GetTestSnapshot();
            var contract = new SmartContractWrapper<INep17ContractABI>(snapshot, testWallet)
            .WithName("GasToken")
            .WithSignerPrivateKey(testSignerPrivKey)
            .WithSignerPublicKey(testSignerPubKey)
            .Build();

            Assert.AreEqual(contract.Name(), "GasToken");
            Assert.AreEqual(contract.Symbol(), "DVP");
            Assert.IsNotNull(contract.GetABI());

            UInt160.TryParse(testSignerPubKey, out var from);
            UInt160.TryParse(testSignerPubKey, out var to);
            FakeBalance(from, ref snapshot);

            var tx = contract.Transfer(from, to, 10L, null) as Transaction;
            Assert.IsNotNull(tx?.Hash);
            FakeTx(tx, ref snapshot);

            var txWrapper = new TransactionWrapper(tx, snapshot, contract.CurrentWallet, TestBlockchain.TheNeoSystem);
            var txResultAsync = await txWrapper.WaitAsync(CancellationToken.None);
            Assert.IsNotNull(txResultAsync.Transaction);
        }

        [TestMethod]
        public void TestWrapperNullableWalletAndSystem()
        {
            var snapshot = TestBlockchain.GetTestSnapshot();
            
            var contract = new SmartContractWrapper<IDVTContractABI>(snapshot, null!)
            .WithScriptHash("0xb34e1025391e953a918231df11478ec21b039e5f")
            .Build();

            Assert.IsNotNull(contract);
            Assert.AreEqual("DvitaToken", contract.Name());
        }

        [TestMethod]
        public void TestWrapperNullableScriptHash()
        {
            var snapshot = TestBlockchain.GetTestSnapshot();
            
            Assert.ThrowsException<ArgumentNullException>(() => {
                var contract = new SmartContractWrapper<IDVTContractABI>(snapshot, null!)
                .WithName("Bare")
                .Build();
            });
        }
    }
}
