import { ApiHideProperty, ApiProperty, ApiPropertyOptional } from "@nestjs/swagger";
import { BlockchainSmartContract } from "../../blockchain/types/blockchain.smart.contract";
import { BlockchainAssetMetadata } from "./blockchain.asset.metadata";

export class BlockchainAssetDto {
    private static readonly DVT_HASH = '0xb34e1025391e953a918231df11478ec21b039e5f';
    private static readonly DVP_HASH = '0xd2a4cff31913016155e38e474a2c06d08be276cf';

    private static readonly NEO_HASH = '0xef4073a0f2b305a38ec4050e4d3d28bc40ea63f5';
    private static readonly GAS_HASH = '0xd2a4cff31913016155e38e474a2c06d08be276cf';

    @ApiHideProperty()
    public static readonly DVT_ASSET = new BlockchainAssetDto(
        BlockchainAssetDto.DVT_HASH,
        'DVT',
        'DVT',
        0,
        new BlockchainAssetMetadata(
            'https://dvita.io/assets/logos/logo2.svg',
            'https://dvita.io/assets/logos/logo2.svg',
            'Dvita is a platform that supports DLT to improve financial inclusion, transparency and the UN’s SDGs.',
            'https://dvita.io/',
            'https://dvita.io/blocks'
        )
    );

    @ApiHideProperty()
    public static readonly DVP_ASSET = new BlockchainAssetDto(
        BlockchainAssetDto.DVP_HASH,
        'DVP',
        'DVP',
        8,
        new BlockchainAssetMetadata(
            'https://dvita.io/assets/logos/logo2.svg',
            'https://dvita.io/assets/logos/logo2.svg',
            'Dvita is a platform that supports DLT to improve financial inclusion, transparency and the UN’s SDGs.',
            'https://dvita.io/',
            'https://dvita.io/blocks'
        )
    );

    @ApiHideProperty()
    public static readonly NEO_ASSET = new BlockchainAssetDto(
        BlockchainAssetDto.NEO_HASH,
        'NEO',
        'NEO',
        0,
        new BlockchainAssetMetadata(
            'https://dvita.io/assets/logos/logo2.svg',
            'https://dvita.io/assets/logos/logo2.svg',
            'Neo is unique in that it was the first public blockchain platform to adopt a dual token mechanism. It separates the rights of governance from the rights of using the network. It also provides a means of acquiring the tokens needing to pay transaction fees.',
            'https://neo.org/',
            'https://neo3.neotube.io/'
        )
    );

    @ApiHideProperty()
    public static readonly GAS_ASSET = new BlockchainAssetDto(
        BlockchainAssetDto.GAS_HASH,
        'GAS',
        'GAS',
        8,
        new BlockchainAssetMetadata(
            'https://dvita.io/assets/logos/logo2.svg',
            'https://dvita.io/assets/logos/logo2.svg',
            'The Neo network charges GAS for the operation and storage of tokens and smart contracts, preventing the abuse of node resources.',
            'https://neo.org/',
            'https://neo3.neotube.io/'
        )
    );

    @ApiProperty()
    readonly hash: string;

    @ApiProperty()
    /*TODO: readonly*/ code: string;

    @ApiPropertyOptional()
    /*TODO: readonly*/ name?: string;

    @ApiPropertyOptional()
    /*TODO: readonly*/ decimals?: string;

    @ApiPropertyOptional()
    metadata?: BlockchainAssetMetadata | undefined;

    constructor(
        hash: string,
        code: string,
        name: string,
        decimals: string  | number | undefined,
        metadata: BlockchainAssetMetadata | undefined
    ) {
        this.hash = hash;
        this.code = code;
        this.name = name;
        this.decimals = decimals?.toString();
        this.metadata = metadata;
    }

    static fromSmartContract(sc: BlockchainSmartContract): BlockchainAssetDto {
        return new BlockchainAssetDto(
            sc.scriptHash,
            sc.code,
            sc.name,
            sc.token?.decimals ?? 0,
            new BlockchainAssetMetadata(sc.token?.iconUrl, sc.token?.iconUrl, sc.description)
        );
    }

    static fromCodeOrHash(codeOrHash: string): BlockchainAssetDto {

        //
        // DVT
        //

        if (codeOrHash.toLowerCase() === BlockchainAssetDto.DVT_HASH.toLowerCase()) {
            return BlockchainAssetDto.DVT_ASSET;
        }

        if (codeOrHash.toUpperCase() === 'DVT') {
            return BlockchainAssetDto.DVP_ASSET;
        }

        //
        // DVP
        //

        if (codeOrHash.toLowerCase() === BlockchainAssetDto.DVP_HASH.toLowerCase()) {
            return BlockchainAssetDto.DVP_ASSET;
        }

        if (codeOrHash.toUpperCase() === 'DVP') {
            return BlockchainAssetDto.DVP_ASSET;
        }

        //
        // NEO
        //

        if (codeOrHash.toLowerCase() === BlockchainAssetDto.NEO_HASH.toLowerCase()) {
            return BlockchainAssetDto.NEO_ASSET;
        }

        if (codeOrHash.toUpperCase() === 'NEO') {
            return BlockchainAssetDto.NEO_ASSET;
        }

        //
        // GAS
        //

        if (codeOrHash.toLowerCase() === BlockchainAssetDto.GAS_HASH.toLowerCase()) {
            return BlockchainAssetDto.GAS_ASSET;
        }

        if (codeOrHash.toUpperCase() === 'GAS') {
            return BlockchainAssetDto.GAS_ASSET;
        }

        // Other by code or hash
        if (codeOrHash.toLowerCase().startsWith('0x')) {
            // By hash
            return new BlockchainAssetDto(codeOrHash, codeOrHash, codeOrHash, 0, undefined);
        } else {
            // By code
            return new BlockchainAssetDto(undefined, codeOrHash, codeOrHash, 0, undefined);
        }
    }
}