import { Decimal } from 'decimal.js';
import { utils } from 'ethers';
import { WalletProvider, BalanceInfo, WalletBalance, WalletInfo } from './types';

export class WalletProviderMetamask implements WalletProvider {
    async connectWallet(): Promise<WalletInfo> {
        const ethereum = await this.ethereumPromise();
        const accounts: string[] = await ethereum.request({ method: 'eth_requestAccounts' });
        const walletBalance: WalletBalance = {};
        for (const account of accounts) {
            const balanceHex: string = await ethereum.request({ method: 'eth_getBalance', params:[account, 'latest']});
            const balance = new Decimal(balanceHex).div(new Decimal(10).pow(18)).toString();
            walletBalance[account] = {
                amount: balance,
                symbol: 'ETH',
                icon: 'assets/icons/ethereum-large.svg',
            };
        }
        return {
            account: {
                address: accounts[0],
                label: 'my wallet',
            },
            balance: walletBalance,
            providerName: 'metamask',
        }
    }

    async transfer(from: string, to: string, amount: string): Promise<string> {
        const ethereum = await this.ethereumPromise();
        return ethereum.request({
            method: 'eth_sendTransaction',
            params: [{
                from: from,
                to: to,
                value: amount,
                gasPrice: '0x09184e72a000', // TODO fetch fee
                gas: '0x2710', // TODO fetch fee
            }],
        });
    }

    async sign(message: string) {
        const ethereum = await this.ethereumPromise();
        const toSign = utils.hexlify(utils.toUtf8Bytes(message));
        const accounts: string[] = await ethereum.request({ method: 'eth_requestAccounts' });
        return ethereum.request({ method: 'personal_sign', params:[accounts[0], toSign]});
    }

    private ethereumPromise(): Promise<any> {
        return new Promise(resolve => {
            const ethereum = (window as any).ethereum;
            const hasMetaMask = ethereum && ethereum.isMetaMask;
            if (hasMetaMask) {
                resolve(ethereum);
            } else {
                throw new Error('MetaMask not detected');
            }
        })
    }
}
