import { WalletProvider, WalletInfo, AccountInfo, WalletBalance } from './types';

export class WalletProviderDvita implements WalletProvider {
    private dVITAWalletPromise: Promise<dVITAWallet>;

    constructor() {
        this.dVITAWalletPromise = new Promise(resolve => {
            window.addEventListener('dVITAWallet.EVENT.READY', () => {
                resolve(new (window as any).dVITAWallet.Init());
            });
        });
    }

    async connectWallet(): Promise<WalletInfo> {
        const wallet = await this.dVITAWalletPromise;
        const account = await wallet.getAccount();
        const balance = await wallet.getBalance().catch(err => {
            console.error(err);
            return { [account.address]: [{ symbol: 'DVG', amount: '0', contract: '' }] };
        });
        return {
            account: account,
            balance: dvgBalance(balance),
            providerName: 'dvita',
        };
    }

    transfer(from: string, to: string, amount: string): Promise<string> {
        throw new Error('TODO implement whenever OpenAPI is reachable');
    }

    async sign(messageToSign: string): Promise<{ publicKey: string, salt: string, data: string }> {
        const wallet = await this.dVITAWalletPromise;
        const { message, ...result } = await wallet.signMessage({ message: messageToSign });
        return result;
    }
}

interface Balance {
    amount: string,
    symbol: string,
    contract: string
}

interface dVITAWallet {
    getProvider: () => Promise<any>;
    getNetworks: () => Promise<any>;
    getAccount: () => Promise<AccountInfo>;
    getPublicKey: () => Promise<any>;
    getBalance: () => Promise<Record<string, Balance[]>>;
    getStorage: () => Promise<any>;
    invokeRead: () => Promise<any>;
    invokeReadMulti: () => Promise<any>;
    verifyMessage: () => Promise<any>;
    getTransaction: () => Promise<any>;
    invoke: () => Promise<any>;
    invokeMulti: () => Promise<any>;
    signMessage: (props: { message: string }) => Promise<{ message: string, publicKey: string, salt: string, data: string }>;
    deploy: () => Promise<any>;
    send: () => Promise<any>;
    getBlock: () => Promise<any>;
}

function dvgBalance(balances: Record<string, Balance[]>): WalletBalance {
    const out: WalletBalance = {};
    for (const address in balances) {
        const gasBalance = balances[address].find(acc => acc.symbol === 'DVG');
        out[address] = {
            ...(gasBalance ? gasBalance : { symbol: 'DVG', amount: '0' }),
            icon: 'assets/icons/token_dvita.png',
        };
    }
    return out;
}
