import { HttpClient } from '@angular/common/http';
import { Injectable } from "@angular/core";
import { Router } from '@angular/router';
import { BehaviorSubject } from 'rxjs';
import { Logger } from './../../logger/logger';
import { WalletProviderName } from './../wallet/types';
import { WalletPluginService } from './../wallet/wallet.plugin.service';
import { UserInfoService } from './user.info.service';
import { UserService } from './user.service';

@Injectable({ providedIn: 'root' })
export class UserAuthService {

    public readonly isLoggedIn: BehaviorSubject<boolean> = new BehaviorSubject<boolean>(false);

    private readonly logger = new Logger(UserAuthService.name);

    constructor(
        private readonly userService: UserService,
        private readonly userInfoService: UserInfoService,
        private readonly router: Router,
        private readonly httpClient: HttpClient,
        private readonly walletPluginService: WalletPluginService
    ) {
        this.authenticateUser();
    }

    public getAuthorizationStatus(): boolean {
        return this.isLoggedIn.getValue();
    }

    public async authorizeUser(provider: WalletProviderName): Promise<void> {
        const wallet = await this.walletPluginService.connectWallet(provider);
        const userInfo = await this.userService.loginWithWallet(wallet);

        this.userInfoService.setUserInfo(userInfo);
        this.logger.debug(`User logged-in: ${JSON.stringify(userInfo)}`);
        this.logIn();
    }

    public logIn(): void {
        this.isLoggedIn.next(true);
    }

    public logOut(): void {
        this.userService.logout();
        this.isLoggedIn.next(false);
    }

    public async signToken(provider: WalletProviderName) {
        const { signable, prefix } = await this.httpClient.get<{ signable: string, prefix: string }>(`/auth/token`).toPromise();
        const messageToSign = prefix + JSON.parse(atob(signable.split('.')[1])).nonce;
        const signature = await this.walletPluginService.sign(provider, messageToSign);
        const header = base64WithoutPadding(JSON.stringify({ alg: 'none' }));
        const payload = base64WithoutPadding(JSON.stringify({ signable: signable, signature: [provider, signature] }));
        const token = `${header}.${payload}.`;
        const result = await this.httpClient.get(`/auth/token/test`, {
            headers: {
                Authorization: `Bearer ${token}`,
            },
            responseType: 'text',
        }).toPromise();
        console.log(result);
    }

    private async authenticateUser(): Promise<void> {
        try {
            const loggedInUserInfo = await this.userService.getLoggedInUser();

            if (loggedInUserInfo) {
                this.userInfoService.setUserInfo(loggedInUserInfo);
                this.logIn();

                this.logger.debug(`User logged-in previously: ${JSON.stringify(loggedInUserInfo)}`);
            } else {
                this.logger.debug(`New user session...`);
            }
        } catch (error) {
            this.logger.error(error);
        }
    }
}

function base64WithoutPadding(str: string) {
    return btoa(str).replace(/=/g, '');
}
