import { SignerWithAddress } from '@nomiclabs/hardhat-ethers/signers';
import { ethers, network, upgrades } from 'hardhat';
import { expect, assert } from 'chai';
import { BigNumber} from 'ethers';

import Web3 from 'web3';
// @ts-ignore
const web3 = new Web3(network.provider) as Web3;

import { TantalisNFT, TantalisNFTV2 } from '../typechain';

let nft: TantalisNFT;
let owner: SignerWithAddress;
let user0: SignerWithAddress;
let user1: SignerWithAddress;
let user2: SignerWithAddress;

let NFTName: String = 'TantalisNFT';
let NFTSymbol: String = 'TNFT';

describe('Tantalis NFT', () => {
    beforeEach(async () => {
        [owner, user0, user1, user2] = await ethers.getSigners();
        const TantalisNFT = await ethers.getContractFactory('TantalisNFT');
		nft = await upgrades.deployProxy(TantalisNFT, [NFTName, NFTSymbol, owner.address, [user2.address], ['https://']]) as TantalisNFT;
    });
    
	describe('Deployment', () => {
        it('Check basic namings', async () => {
            expect(await nft.name()).to.equal(NFTName);
            expect(await nft.symbol()).to.equal(NFTSymbol);
        });
        it('Check Roles', async () => {
            expect(await nft.hasRole(await nft.DEFAULT_ADMIN_ROLE(), owner.address)).to.be.true;
            expect(await nft.hasRole(await nft.ADMIN_ROLE(), owner.address)).to.be.true;
            expect(await nft.hasRole(await nft.PAUSER_ROLE(), owner.address)).to.be.true;
            expect(await nft.hasRole(await nft.UPGRADER_ROLE(), owner.address)).to.be.true;
        });
    });

	describe('Functionalities', () => {
		it('Create NFT', async () => {
			const tx = await nft.connect(user0).createToken(user0.address, 'https://test');
			const events = (await tx.wait()).events?.find((event: any) => event.event === 'Transfer');

			expect(await nft.ownerOf(1)).to.equal(user0.address).to.equal(events?.args!.to);
			expect(await nft.tokenURI(1)).to.equal('https://test');
			expect(events?.args!.tokenId).to.equal(1);
		});
		it('Create NFT and transferFrom default approved address to another address', async () => {
			await nft.connect(user0).createToken(user0.address, 'https://test');
			await nft.connect(owner).transferFrom(user0.address, user2.address, 1);
			
			expect(await nft.ownerOf(1)).to.equal(user2.address);
		});
		it('Create Batch NFTs', async () => {
			const tx = await nft.connect(user0).createBatchTokens([user0.address, user1.address], ['https://test', 'https://test2']);
			const events = (await tx.wait()).events?.filter((event: any) => event.event === 'Transfer') as any;

			expect(await nft.ownerOf(1)).to.equal(user0.address).to.equal(events[0].args!.to);
			expect(await nft.tokenURI(1)).to.equal('https://test');
			expect(events[0].args!.tokenId).to.equal(1);
			expect(await nft.ownerOf(2)).to.equal(user1.address).to.equal(events[1].args!.to);
			expect(await nft.tokenURI(2)).to.equal('https://test2');
			expect(events[1].args!.tokenId).to.equal(2);
		});
		it('Create Batch NFTs and transferFrom default approved address to another address', async () => {
			await nft.connect(user0).createBatchTokens([user0.address, user1.address], ['https://test', 'https://test2'])
			await nft.connect(owner).transferFrom(user0.address, user2.address, 1);

			expect(await nft.ownerOf(1)).to.equal(user2.address);
		});
		it('Cannot create Batch NFTs without any URI', async () => {
			await expect(nft.connect(user0).createBatchTokens([user0.address, user1.address], []))
            .to.revertedWith('You need to pass at least one token URI');
		});
		it('Cannot create Batch NFTs without any receivers', async () => {
			await expect(nft.connect(user0).createBatchTokens([], ['https://test', 'https://test2']))
            .to.revertedWith('Users array length must be equal to tokens URI array length');
		});
		it('Cannot create Batch NFTs with not equal receivers array and tokens URI array', async () => {
			await expect(nft.connect(user0).createBatchTokens([user0.address], ['https://test', 'https://test2']))
            .to.revertedWith('Users array length must be equal to tokens URI array length');
		});
		it('Transfer NFT', async () => {
			const tx = await nft.connect(user2).transferFrom(user2.address, user0.address, 0);
			const events = (await tx.wait()).events?.find((event: any) => event.event === 'Transfer');

			expect(await nft.ownerOf(0)).to.equal(user0.address).to.equal(events?.args!.to);
			expect(events?.args!.from).to.equal(user2.address);
			expect(events?.args!.tokenId).to.equal(0);
		});
		it('Batch tramsfer of NFTs', async () => {
			await nft.connect(user1).createBatchTokens(
				[user0.address, user1.address, user2.address, user0.address],
				['https://test', 'https://test2', 'https://test3', 'https://test4']
			);
			await nft.connect(user0).safeBatchTransferFrom(user0.address, user1.address, [1,4]);
			await nft.connect(user2).safeBatchTransferFrom(user2.address, user1.address, [0,3]);

			expect(await nft.ownerOf(0)).to.equal(user1.address);
			expect(await nft.ownerOf(1)).to.equal(user1.address);
			expect(await nft.ownerOf(2)).to.equal(user1.address);
			expect(await nft.ownerOf(3)).to.equal(user1.address);
			expect(await nft.ownerOf(4)).to.equal(user1.address);
		});
		it('Batch tramsfer of NFTs by approval for all account', async () => {
			await nft.connect(user1).createBatchTokens(
				[user0.address, user1.address, user2.address, user0.address],
				['https://test', 'https://test2', 'https://test3', 'https://test4']
			);
			await nft.connect(user0).setApprovalForAll(user1.address, true);
			await nft.connect(user2).setApprovalForAll(user1.address, true);
			await nft.connect(user1).safeBatchTransferFrom(user0.address, user1.address, [1,4]);
			await nft.connect(user1).safeBatchTransferFrom(user2.address, user1.address, [0,3]);
		
			expect(await nft.ownerOf(0)).to.equal(user1.address);
			expect(await nft.ownerOf(1)).to.equal(user1.address);
			expect(await nft.ownerOf(2)).to.equal(user1.address);
			expect(await nft.ownerOf(3)).to.equal(user1.address);
			expect(await nft.ownerOf(4)).to.equal(user1.address);
		});
		it('Cannot batch tramsfer of NFTs by not approved or non owner of that nfts', async () => {
			await nft.connect(user1).createBatchTokens(
				[user0.address, user1.address, user2.address, user0.address],
				['https://test', 'https://test2', 'https://test3', 'https://test4']
			);
			
			await expect(nft.connect(user1).safeBatchTransferFrom(user0.address, user2.address, [0,1,2,3,4]))
            .to.revertedWith('Transfer caller is not owner nor approved for all tokens');
		});
		it('Cannot batch tramsfer of NFTs by non owner of all nfts', async () => {
			await nft.connect(user1).createBatchTokens(
				[user1.address, user0.address, user2.address, user0.address],
				['https://test', 'https://test2', 'https://test3', 'https://test4']
			);
			
			await expect(nft.connect(user1).safeBatchTransferFrom(user1.address, user2.address, [1,2,0,3,4]))
            .to.revertedWith('ERC721: transfer from incorrect owner');
			expect(await nft.ownerOf(0)).to.equal(user2.address);
			expect(await nft.ownerOf(1)).to.equal(user1.address);
			expect(await nft.ownerOf(2)).to.equal(user0.address);
			expect(await nft.ownerOf(3)).to.equal(user2.address);
			expect(await nft.ownerOf(4)).to.equal(user0.address);
		});
		it('Cannot batch tramsfer of NFTs by not approved of all nfts', async () => {
			await nft.connect(user1).createBatchTokens(
				[user0.address, user1.address, user2.address, user0.address],
				['https://test', 'https://test2', 'https://test3', 'https://test4']
			);
			await nft.connect(user1).setApprovalForAll(user2.address, true);

			await expect(nft.connect(user2).safeBatchTransferFrom(user1.address, user2.address, [1,2,3,4]))
            .to.revertedWith('ERC721: transfer from incorrect owner');
			expect(await nft.ownerOf(1)).to.equal(user0.address);
			expect(await nft.ownerOf(2)).to.equal(user1.address);
			expect(await nft.ownerOf(3)).to.equal(user2.address);
			expect(await nft.ownerOf(4)).to.equal(user0.address);
		});
		it('Cannot batch tramsfer of NFTs by not approved or non owner of all nfts and not existing token', async () => {			
			await expect(nft.connect(user1).safeBatchTransferFrom(user1.address, user2.address, []))
            .to.revertedWith('You need to pass at least one token to transfer');
		});
		it('Cannot call any transferring function (even creating functions) when contract paused', async () => {
			await nft.pause();

			await expect(nft.connect(user1).createToken(user1.address, 'someMetaData'))
            .to.revertedWith('Pausable: paused');
			await expect(nft.connect(user1).createBatchTokens([user0.address, user1.address, user2.address], ['someMetaData', 'anotherMetaData', 'Sometext']))
            .to.revertedWith('Pausable: paused');
			await expect(nft.connect(user2).transferFrom(user2.address, user0.address, 0))
            .to.revertedWith('Pausable: paused');
		});
		it('Upgrade while contract paused', async () => {
			await nft.pause();
			const TantalisNFTV2 = await ethers.getContractFactory('TantalisNFTV2');
        	const nftV2 = await upgrades.upgradeProxy(nft.address, TantalisNFTV2) as TantalisNFTV2;

			expect(await nftV2.version()).to.equal(BigNumber.from(2));
			expect(await nftV2.address).to.equal(nft.address);
			expect(await nftV2.ownerOf(0)).to.equal(user2.address);
		});
		it('Pause and unpause', async () => {
			await nft.pause();
			await nft.unpause();
			
			const tx = await nft.connect(user0).createToken(user0.address, 'https://test');
			const events = (await tx.wait()).events?.find((event: any) => event.event === 'Transfer');

			expect(await nft.ownerOf(1)).to.equal(user0.address).to.equal(events?.args!.to);
			expect(await nft.tokenURI(1)).to.equal('https://test');
			expect(events?.args!.tokenId).to.equal(1);
		});
		it('Cannot call any transferring function (even creating functions) when user is in blocked list', async () => {
			await nft.connect(user1).createToken(user1.address, 'https://test');
			await nft.blockListUpdate(user1.address, true);

			await expect(nft.connect(user1).createToken(user1.address, 'someMetaData'))
            .to.revertedWith('Token receiver is in the block list');
			await expect(nft.connect(user1).createBatchTokens([user0.address, user1.address, user2.address], ['someMetaData', 'anotherMetaData', 'Sometext']))
            .to.revertedWith('Token receiver is in the block list');
			expect(await nft.ownerOf(1)).to.equal(user1.address);
			await expect(nft.connect(user1).transferFrom(user1.address, user0.address, 1))
            .to.revertedWith('Token sender is in the block list');
		});
		it('User can continue to transfer NFTs after unlisting from blocked list', async () => {
			await nft.blockListUpdate(user1.address, true);
			await nft.blockListUpdate(user1.address, false);

			await nft.connect(user1).createToken(user1.address, 'https://test');
			expect(await nft.ownerOf(1)).to.equal(user1.address);
		});
	});
});
