import { SignerWithAddress } from '@nomiclabs/hardhat-ethers/signers';
import { ethers, network, upgrades } from 'hardhat';
import { expect, assert } from 'chai';
import { BigNumber} from 'ethers';
import { solidityKeccak256, arrayify } from "ethers/lib/utils";

import Web3 from 'web3';

// ToDo:: merge nft and marketplace contracts
// @ts-ignore
const web3 = new Web3(network.provider) as Web3;

import { Marketplace, MarketplaceV2, TestNFT } from '../typechain';

let marketplace: Marketplace;
let testNFT: TestNFT;
let owner: SignerWithAddress;
let executor: SignerWithAddress;
let validator: SignerWithAddress;
let user0: SignerWithAddress;
let user1: SignerWithAddress;
let user2: SignerWithAddress;
let users: Array<SignerWithAddress>;

const price: BigNumber = BigNumber.from(10).pow(18).mul(10);
let NFTName: String = 'TantalisNFT';
let NFTSymbol: String = 'TNFT';

describe('Marketplace', () => {
    beforeEach(async () => {
        [owner, executor, validator, user0, user1, user2, ...users] = await ethers.getSigners();
        const Marketplace = await ethers.getContractFactory('Marketplace');
		marketplace = await upgrades.deployProxy(Marketplace, [validator.address, executor.address]) as Marketplace;
		await marketplace.deployed();

        const TestNFT = await ethers.getContractFactory('TestNFT');
        testNFT = await upgrades.deployProxy(TestNFT, [NFTName, NFTSymbol, owner.address, [user0.address], ['https://']]) as TestNFT;
        await testNFT.deployed();

        await testNFT.connect(user0).setApprovalForAll(marketplace.address, true);
    });
    describe('Deployment', () => {
        it('Check Roles', async () => {
            expect(await marketplace.hasRole(await marketplace.DEFAULT_ADMIN_ROLE(), owner.address)).to.be.true;
            expect(await marketplace.hasRole(await marketplace.ADMIN_ROLE(), owner.address)).to.be.true;
            expect(await marketplace.hasRole(await marketplace.PAUSER_ROLE(), owner.address)).to.be.true;
            expect(await marketplace.hasRole(await marketplace.UPGRADER_ROLE(), owner.address)).to.be.true;
            expect(await marketplace.hasRole(await marketplace.VALIDATOR_ROLE(), validator.address)).to.be.true;
            expect(await marketplace.hasRole(await marketplace.EXECUTOR_ROLE(), executor.address)).to.be.true;
        });
    });
	describe('Functions', () => {
        it('Buy and send royalties', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                )
            );

            testNFT.setDefaultRoyalty(owner.address, 1000);

            const signature = await validator.signMessage(message);
            
            await marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature, {value: price.div(10)});

            expect(await testNFT.ownerOf(0)).to.equal(user1.address);
        });
        it('Not an executor cannot buy', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                )
            );

            const signature = await validator.signMessage(message);

            expect(marketplace.connect(user1).buy(user0.address, user1.address, testNFT.address, 0, price, signature))
            .to.be.revertedWith("You should have an executor role");
        });
        it('Cannot buy without validator sign', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                )
            );

            const signature = await user1.signMessage(message);

            await expect(marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature))
            .to.be.revertedWith("Validator address is not correct");
        });
        it('Export NFT', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, 0]
                )
            );

            const signature = await validator.signMessage(message);
            
            await marketplace.connect(executor).exportNFT(user0.address, user1.address, testNFT.address, 0, 0, signature);

            expect(await testNFT.ownerOf(0)).to.equal(user1.address);
        });
        it('Not an executor cannot export NFT', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, 1]
                )
            );

            const signature = await validator.signMessage(message);

            expect(marketplace.connect(user1).buy(user0.address, user1.address, testNFT.address, 0, 1, signature))
            .to.be.revertedWith("You should have an executor role");
        });
        it('Cannot export NFT without validator sign', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, 2]
                )
            );

            const signature = await user1.signMessage(message);

            await expect(marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, 2, signature))
            .to.be.revertedWith("Validator address is not correct");
        });
        it('Cannot use the nonce twice in the export NFT', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, 0]
                )
            );

            const signature = await validator.signMessage(message);
            
            await marketplace.connect(executor).exportNFT(user0.address, user1.address, testNFT.address, 0, 0, signature);
            await testNFT.connect(user1).transferFrom(user1.address, user0.address, 0);
            await testNFT.connect(user1).setApprovalForAll(marketplace.address, true);

            await expect(marketplace.connect(executor).exportNFT(user0.address, user1.address, testNFT.address, 0, 0, signature))
            .to.be.revertedWith("nonce has been used before");
        });
        it('Upgrade while contract paused', async () => {
			await marketplace.pause();
			const MarketplaceV2 = await ethers.getContractFactory('MarketplaceV2');
        	const marketplaceV2 = await upgrades.upgradeProxy(marketplace.address, MarketplaceV2) as MarketplaceV2;
			await marketplace.unpause();

			expect(await marketplaceV2.version()).to.equal(BigNumber.from(2));
			expect(await marketplaceV2.address).to.equal(marketplace.address);
			expect(await marketplaceV2.hasRole(await marketplace.VALIDATOR_ROLE(), validator.address)).to.be.true;
		});
        it('Pause and unpause', async () => {
			const message = arrayify(
                solidityKeccak256( 
                    ["address", "address", "address", "address", "uint256", "uint256"], 
                    [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                )
            );

            const signature = await validator.signMessage(message);

			await marketplace.pause();
			expect(await marketplace.paused()).to.be.true;
            await expect(marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature))
            .to.be.revertedWith("Pausable: paused");

			await marketplace.unpause();
			await marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature);
            expect(await testNFT.ownerOf(0)).to.equal(user1.address);
		});
        describe('Multi buying process', () => {
            beforeEach(async () => {
                await testNFT.connect(user1).setApprovalForAll(marketplace.address, true);
                await testNFT.connect(user2).setApprovalForAll(marketplace.address, true);
                await testNFT.connect(user1).createToken(user1.address, 'https://test');
                await testNFT.connect(user1).createToken(user1.address, 'https://test');
                await testNFT.connect(user1).createToken(user1.address, 'https://test');
                await testNFT.connect(user1).createToken(user1.address, 'https://test');
                await testNFT.connect(user1).createToken(user1.address, 'https://test');
                await testNFT.connect(user2).createToken(user2.address, 'https://test');
            });
            it('Sell/Buy one NFT many times', async () => {
                const message = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                    )
                );
    
                const message2 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, user2.address, testNFT.address, 0, price.mul(15).div(10)]
                    )
                );
    
                const message3 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user2.address, users[0].address, testNFT.address, 0, price.mul(5).div(10)]
                    )
                );
    
                const signature = await validator.signMessage(message);
                const signature2 = await validator.signMessage(message2);
                const signature3 = await validator.signMessage(message3);
                
                await marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature);
                await marketplace.connect(executor).buy(user1.address, user2.address, testNFT.address, 0, price.mul(15).div(10), signature2);
                await marketplace.connect(executor).buy(user2.address, users[0].address, testNFT.address, 0, price.mul(5).div(10), signature3);
    
                expect(await testNFT.ownerOf(0)).to.equal(users[0].address);
            });
            it('Selling many NFTs', async () => {
                const message = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                    )
                );
    
                const message2 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, user2.address, testNFT.address, 1, price.mul(15).div(10)]
                    )
                );
    
                const message3 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, users[0].address, testNFT.address, 4, price.mul(5).div(10)]
                    )
                );
    
                const message4 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, users[1].address, testNFT.address, 5, price]
                    )
                );
    
                
                const message5 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user2.address, users[2].address, testNFT.address, 6, price]
                    )
                );
    
                const signature = await validator.signMessage(message);
                const signature2 = await validator.signMessage(message2);
                const signature3 = await validator.signMessage(message3);
                const signature4 = await validator.signMessage(message4);
                const signature5 = await validator.signMessage(message5);
                
                await marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature);
                await marketplace.connect(executor).buy(user1.address, user2.address, testNFT.address, 1, price.mul(15).div(10), signature2);
                await marketplace.connect(executor).buy(user1.address, users[0].address, testNFT.address, 4, price.mul(5).div(10), signature3);
                await marketplace.connect(executor).buy(user1.address, users[1].address, testNFT.address, 5, price, signature4);
                await marketplace.connect(executor).buy(user2.address, users[2].address, testNFT.address, 6, price, signature5);
    
                expect(await testNFT.ownerOf(0)).to.equal(user1.address);
                expect(await testNFT.ownerOf(1)).to.equal(user2.address);
                expect(await testNFT.ownerOf(4)).to.equal(users[0].address);
                expect(await testNFT.ownerOf(5)).to.equal(users[1].address);
                expect(await testNFT.ownerOf(6)).to.equal(users[2].address);
            });
            it('Pause/unpause while selling many NFTs', async () => {
                const message = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user0.address, user1.address, testNFT.address, 0, price]
                    )
                );
    
                const message2 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, user2.address, testNFT.address, 1, price.mul(15).div(10)]
                    )
                );
    
                const message3 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, users[0].address, testNFT.address, 4, price.mul(5).div(10)]
                    )
                );
    
                const message4 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user1.address, users[1].address, testNFT.address, 5, price]
                    )
                );
    
                
                const message5 = arrayify(
                    solidityKeccak256( 
                        ["address", "address", "address", "address", "uint256", "uint256"], 
                        [executor.address, user2.address, users[2].address, testNFT.address, 6, price]
                    )
                );
    
                const signature = await validator.signMessage(message);
                const signature2 = await validator.signMessage(message2);
                const signature3 = await validator.signMessage(message3);
                const signature4 = await validator.signMessage(message4);
                const signature5 = await validator.signMessage(message5);
                
                await marketplace.connect(executor).buy(user0.address, user1.address, testNFT.address, 0, price, signature);
                await marketplace.connect(executor).buy(user1.address, user2.address, testNFT.address, 1, price.mul(15).div(10), signature2);           
                await marketplace.pause();
                
                expect(await marketplace.paused()).to.be.true;
                await expect(marketplace.connect(executor).buy(user1.address, users[0].address, testNFT.address, 4, price.mul(5).div(10), signature3))
                .to.be.revertedWith("Pausable: paused");
    
                await marketplace.unpause();
                await marketplace.connect(executor).buy(user1.address, users[0].address, testNFT.address, 4, price.mul(5).div(10), signature3);
                await marketplace.connect(executor).buy(user1.address, users[1].address, testNFT.address, 5, price, signature4);
                await marketplace.connect(executor).buy(user2.address, users[2].address, testNFT.address, 6, price, signature5);
    
                expect(await testNFT.ownerOf(0)).to.equal(user1.address);
                expect(await testNFT.ownerOf(1)).to.equal(user2.address);
                expect(await testNFT.ownerOf(4)).to.equal(users[0].address);
                expect(await testNFT.ownerOf(5)).to.equal(users[1].address);
                expect(await testNFT.ownerOf(6)).to.equal(users[2].address);
            });
        });
    });
});
