// SPDX-License-Identifier: MIT
pragma solidity 0.8.15;

import "@openzeppelin/contracts-upgradeable/token/ERC721/extensions/ERC721URIStorageUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/token/common/ERC2981Upgradeable.sol";
import "@openzeppelin/contracts-upgradeable/access/AccessControlUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/security/ReentrancyGuardUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/security/PausableUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/proxy/utils/Initializable.sol";
import "@openzeppelin/contracts-upgradeable/proxy/utils/UUPSUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/utils/CountersUpgradeable.sol";

/**
* Terms and Conditions of Use
* No intellectual property rights are being transferred to the NFT buyer. More specifically, the buyer obtains none of the following rights:
* - No Copyright
* - No Commercialization Rights
* - No Derivative Rights
* - No Rights to Claim Any Royalties upon Resale of the NFT
* The NFT buyer has only the following rights:
* - Right to Have, Hold, Admire and Display the NFT in Private, Non-Commercial Settings
* - Right to Resell the NFT Acquired, Subject to These Same Terms and Conditions
*/

contract TestNFT is Initializable, ERC721URIStorageUpgradeable, ERC2981Upgradeable, PausableUpgradeable, AccessControlUpgradeable, ReentrancyGuardUpgradeable, UUPSUpgradeable {
    using CountersUpgradeable for CountersUpgradeable.Counter;
    CountersUpgradeable.Counter private _tokenIdCounter;

    address public marketplaceAddress;
    string public termsAndConditions;
    bytes32 public constant ADMIN_ROLE = keccak256("ADMIN_ROLE");
    bytes32 public constant PAUSER_ROLE = keccak256("PAUSER_ROLE");
    bytes32 public constant UPGRADER_ROLE = keccak256("UPGRADER_ROLE");

    mapping(address => bool) public blockList;

    /**
    * @notice Event firing whenever blocklist updates
    * @param user user address
    */
    event BlockListUpdated(address indexed user, bool value);
    
    /**
    * @dev Need to be initialized. 
    * For more information: https://docs.openzeppelin.com/upgrades-plugins/1.x/writing-upgradeable#initializing_the_implementation_contract
    * @custom:oz-upgrades-unsafe-allow constructor
    */
    constructor() initializer {}

    /**
    * @notice Creating Tantalis NFT
    * @dev Calls only once
    * @param _name name of the token
    * @param _symbol symbol of the token
    * @param _marketplaceAddress address of the nft marketplace
    * @param _to optional param of token receiving addresses
    * @param _tokensURI optional param of tokens metadata
    */
    function initialize(string calldata _name, string calldata _symbol, address _marketplaceAddress, address[] memory _to, string[] memory _tokensURI) initializer public {
        __ERC721_init(_name, _symbol);
        __ERC721URIStorage_init();
        __Pausable_init();
        __AccessControl_init();
        __ReentrancyGuard_init();
        __UUPSUpgradeable_init();
        __ERC2981_init();
        
        _setupRole(DEFAULT_ADMIN_ROLE, msg.sender);
        _setupRole(ADMIN_ROLE, msg.sender);
        _setupRole(PAUSER_ROLE, msg.sender);
        _setupRole(UPGRADER_ROLE, msg.sender);
        _setRoleAdmin(PAUSER_ROLE, ADMIN_ROLE);
        _setRoleAdmin(UPGRADER_ROLE, ADMIN_ROLE);

        marketplaceAddress = _marketplaceAddress;

        if(_to.length > 0) {
            _createBatchTokens(_to, _tokensURI);
        }
        termsAndConditions = "Terms and Conditions of Use \nNo intellectual property rights are being transferred to the NFT buyer. More specifically, the buyer obtains none of the following rights: \n - No Copyright \n - No Commercialization Rights \n - No Derivative Rights \n - No Rights to Claim Any Royalties upon Resale of the NFT \nThe NFT buyer has only the following rights: \n - Right to Have, Hold, Admire and Display the NFT in Private, Non-Commercial Settings \n - Right to Resell the NFT Acquired, Subject to These Same Terms and Conditions";
    }

    /**
    * @notice Pausing function
    * @dev only users with Pauser role can call this function
    */
    function pause() external {
        require(hasRole(PAUSER_ROLE, msg.sender), "You should have a pauser role");

        _pause();
    }

    /**
    * @notice Unpausing function
    * @dev only users with Pauser role can call this function
    */
    function unpause() external {
        require(hasRole(PAUSER_ROLE, msg.sender), "You should have a pauser role");

        _unpause();
    }

    /**
    * @notice Updating users blocklist
    * @dev only users with Admin role can call this function
    * @param _user user address
    * @param _value user address
    */
    function blockListUpdate(address _user, bool _value) external {
        require(hasRole(ADMIN_ROLE, msg.sender), "You should have an admin role");

        blockList[_user] = _value;
        emit BlockListUpdated(_user, _value);
    }

    /**
    * @notice Updating nft marketplace address
    * @dev Default address of marketplace. It is used for approving while creating new token
    * @param _marketplaceAddress address of the nft marketplace
    */
    function updateMarketplaceAddress(address _marketplaceAddress) external {
        require(hasRole(ADMIN_ROLE, msg.sender), "You should have an admin role");
        
        marketplaceAddress = _marketplaceAddress;
    }

    /**
    * @notice Setting default royalties to the nft
    * @param receiver royalty receiver address
    * @param feeNumerator fee numerator
    */
    function setDefaultRoyalty(address receiver, uint96 feeNumerator) external {
        require(hasRole(ADMIN_ROLE, msg.sender), "You should have an admin role");

        _setDefaultRoyalty(receiver, feeNumerator);
    }

    /**
    * @notice Creating new NFT
    * @param _to address where new NFT will be minted
    * @param _tokenURI uri to the metadate or metadate of the token
    * @return newly created token id
    */
    function createToken(address _to, string memory _tokenURI) external nonReentrant returns (uint256) {
        return _createToken(_to, _tokenURI);
    }

    /**
    * @notice Creating batch of new NFTs
    * @param _to addresses where new NFTs will be minted
    * @param _tokensURI uris array for metadata or token metadata array
    * @return newly created array of token ids
    */
    function createBatchTokens(address[] memory _to, string[] memory _tokensURI) external nonReentrant returns (uint256[] memory) {
        require(_tokensURI.length > 0, "You need to pass at least one token URI");

        return _createBatchTokens(_to, _tokensURI);
    }

    /**
    * @notice Batch Transfer of NFTs
    * @param _from address sending from
    * @param _to address sending to
    * @param _ids token ids
    */
    function safeBatchTransferFrom(address _from, address _to, uint256[] memory _ids) external nonReentrant {
        require(_from == msg.sender || isApprovedForAll(_from, msg.sender), "Transfer caller is not owner nor approved for all tokens");
        require(_ids.length > 0, "You need to pass at least one token to transfer");
        
        for(uint256 i = 0; i < _ids.length; i++) {
            _safeTransfer(_from, _to, _ids[i], "");
        }
    }

    function _createToken(address _to, string memory _tokenURI) internal returns (uint256) {
        _safeMint(_to, _tokenIdCounter.current());
        _setTokenURI(_tokenIdCounter.current(), _tokenURI);
        _tokenIdCounter.increment();
        
        if(!isApprovedForAll(marketplaceAddress, _to)){
            _setApprovalForAll(_to, marketplaceAddress, true);
        }

        return _tokenIdCounter.current() - 1;
    }

    function _createBatchTokens(address[] memory _to, string[] memory _tokensURI) internal returns (uint256[] memory) {
        require(_to.length == _tokensURI.length, "Users array length must be equal to tokens URI array length");

        uint256[] memory newlyCreatedTokens = new uint256[](_tokensURI.length);

        for(uint256 i = 0; i < _tokensURI.length; i++) {
            newlyCreatedTokens[i] = _createToken(_to[i], _tokensURI[i]);
        }

        return newlyCreatedTokens;
    }

    function _authorizeUpgrade(address newImplementation)
        internal
        onlyRole(UPGRADER_ROLE)
        override
    {}

    function _beforeTokenTransfer(address _from, address _to, uint256 _tokenId)
        internal
        whenNotPaused
        override
    {
        require (!blockList[_from], "Token sender is in the block list");
        require (!blockList[_to], "Token receiver is in the block list");

        super._beforeTokenTransfer(_from, _to, _tokenId);
    }

    /**
    * @notice The following function override is required for Solidity
    * @dev See {IERC165-supportsInterface}.
    * @param _interfaceId interface id
    */
    function supportsInterface(bytes4 _interfaceId)
        public
        view
        override(ERC721Upgradeable, ERC2981Upgradeable, AccessControlUpgradeable)
    returns (bool)
    {
        return super.supportsInterface(_interfaceId);
    }
}
