// SPDX-License-Identifier: MIT
pragma solidity 0.8.15;

import "@openzeppelin/contracts/utils/cryptography/ECDSA.sol";
import "@openzeppelin/contracts/token/ERC721/IERC721.sol";
import "@openzeppelin/contracts-upgradeable/interfaces/IERC2981Upgradeable.sol";
import "@openzeppelin/contracts-upgradeable/access/AccessControlUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/security/PausableUpgradeable.sol";
import "@openzeppelin/contracts-upgradeable/utils/introspection/IERC165Upgradeable.sol";
import "@openzeppelin/contracts-upgradeable/proxy/utils/Initializable.sol";
import "@openzeppelin/contracts-upgradeable/proxy/utils/UUPSUpgradeable.sol";

contract Marketplace is Initializable, UUPSUpgradeable, AccessControlUpgradeable, PausableUpgradeable {
    bytes32 public constant ADMIN_ROLE = keccak256("ADMIN_ROLE");
    bytes32 public constant VALIDATOR_ROLE = keccak256("VALIDATOR_ROLE");
    bytes32 public constant EXECUTOR_ROLE = keccak256("EXECUTOR_ROLE");
    bytes32 public constant PAUSER_ROLE = keccak256("PAUSER_ROLE");
    bytes32 public constant UPGRADER_ROLE = keccak256("UPGRADER_ROLE");
    
    mapping(uint256 => bool) usedNonce;
    
    /**
    * @notice Event firing whenever buy function executes
    * @param seller seller address
    * @param buyer buyer address 
    * @param token NFT address for sale
    * @param tokenId NFT id for sale
    * @param price price of the sale
    */
    event Bought(
        address seller,
        address buyer,
        address token,
        uint256 tokenId,
        uint256 price
    );

    /**
    * @notice Event firing whenever exportNFT function executes
    * @param sender sender address
    * @param reciever reciever address 
    * @param token NFT address for sale
    * @param tokenId NFT id for sale
    * @param nonce nonce of the message
    */
    event ExportedNFT(
        address sender,
        address reciever,
        address token,
        uint256 tokenId,
        uint256 indexed nonce
    );
    
    /**
    * @dev Need to be initialized. 
    * For more information: https://docs.openzeppelin.com/upgrades-plugins/1.x/writing-upgradeable#initializing_the_implementation_contract
    * @custom:oz-upgrades-unsafe-allow constructor
    */
    constructor() initializer {}

    /**
    * @notice Marketplace contract
    * @dev Calls only once
    */
    function initialize(address _validator, address _executor) initializer public {
        __Pausable_init();
        __UUPSUpgradeable_init();
        __AccessControl_init();
        
        _setupRole(DEFAULT_ADMIN_ROLE, msg.sender);
        _setupRole(ADMIN_ROLE, msg.sender);
        _setupRole(VALIDATOR_ROLE, _validator);
        _setupRole(EXECUTOR_ROLE, _executor);
        _setupRole(PAUSER_ROLE, msg.sender);
        _setupRole(UPGRADER_ROLE, msg.sender);
        _setRoleAdmin(VALIDATOR_ROLE, ADMIN_ROLE);
        _setRoleAdmin(EXECUTOR_ROLE, ADMIN_ROLE);
        _setRoleAdmin(PAUSER_ROLE, ADMIN_ROLE);
        _setRoleAdmin(UPGRADER_ROLE, ADMIN_ROLE);
    }

    /**
    * @notice Transfer nft from seller to buyer
    * @param _seller seller address
    * @param _buyer buyer address
    * @param _token NFT address for sale
    * @param _tokenId NFT id for sale
    * @param _price price of the sale
    * @param _validatorSign validator signature
    */
    function buy(
        address _seller,
        address _buyer,
        address _token,
        uint256 _tokenId,
        uint256 _price,
        bytes calldata _validatorSign
    ) external payable whenNotPaused {
        require(hasRole(EXECUTOR_ROLE, msg.sender), "You should have an executor role");
        
        bytes32 hash = keccak256(
            abi.encodePacked(
                msg.sender,
                _seller,
                _buyer,
                _token,
                _tokenId,
                _price
            )
        );

        uint256 royalties;

        if(IERC165Upgradeable(_token).supportsInterface((type(IERC2981Upgradeable).interfaceId))) {
            address reciever;
            (reciever, royalties) = IERC2981Upgradeable(_token).royaltyInfo(_tokenId, _price);

            if(royalties > 0 && reciever != address(0)) {
                require(msg.value >= royalties, "Sending amount is less than royalties");

                (bool success, ) = reciever.call{value: royalties}("");

                require(success, "Royalty transfer failed");
            }
        }

        if(msg.value > royalties) {
            (bool success, ) = msg.sender.call{value: (msg.value - royalties)}("");

            require(success, "Extra value return transfer failed");
        }

        bytes32 hashedMessage = ECDSA.toEthSignedMessageHash(hash);

        require(
            hasRole(VALIDATOR_ROLE, ECDSA.recover(hashedMessage, _validatorSign)),
            "Validator address is not correct"
        );

        IERC721(_token).safeTransferFrom(_seller, _buyer, _tokenId);

        emit Bought(_seller, _buyer, _token, _tokenId, _price);
    }

    /**
    * @notice Export nft from sender to reciever
    * @param _sender sender address
    * @param _reciever reciever address 
    * @param _token NFT address for sale
    * @param _tokenId NFT id for sale
    * @param _nonce nonce of the message
    * @param _validatorSign validator signature
    */
    function exportNFT(
        address _sender,
        address _reciever,
        address _token,
        uint256 _tokenId,
        uint256 _nonce,
        bytes calldata _validatorSign
    ) external whenNotPaused {
        require(hasRole(EXECUTOR_ROLE, msg.sender), "You should have an executor role");
        require(usedNonce[_nonce] == false, "nonce has been used before");
        
        usedNonce[_nonce] = true;

        bytes32 hash = keccak256(
            abi.encodePacked(
                msg.sender,
                _sender,
                _reciever,
                _token,
                _tokenId,
                _nonce
            )
        );

        bytes32 hashedMessage = ECDSA.toEthSignedMessageHash(hash);

        require(
            hasRole(VALIDATOR_ROLE, ECDSA.recover(hashedMessage, _validatorSign)),
            "Validator address is not correct"
        );

        IERC721(_token).safeTransferFrom(_sender, _reciever, _tokenId);

        emit ExportedNFT(_sender, _reciever, _token, _tokenId, _nonce);
    }

    /**
    * @notice Pausing function
    * @dev only users with Pauser role can call this function
    */
    function pause() public {
        require(hasRole(PAUSER_ROLE, msg.sender), "You should have a pauser role");

        _pause();
    }

    /**
    * @notice Unpausing function
    * @dev only users with Pauser role can call this function
    */
    function unpause() public {
        require(hasRole(PAUSER_ROLE, msg.sender), "You should have a pauser role");

        _unpause();
    }

    /**
    * @notice The following function override is required for Solidity
    * @dev See {IERC165-supportsInterface}.
    * @param _interfaceId interface id
    */
    function supportsInterface(bytes4 _interfaceId)
        public
        view
        override(AccessControlUpgradeable)
    returns (bool)
    {
        return super.supportsInterface(_interfaceId);
    }

    function _authorizeUpgrade(address newImplementation)
        internal
        onlyRole(UPGRADER_ROLE)
        override
    {}
}
