﻿using System;
using System.ComponentModel;
using System.Numerics;
using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;
// using Neo.SmartContract.Framework.Attributes;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.TantalisNFTContract")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA Tantalis NFT")]
    [ContractPermission("*", "*")]
    public class TantalisNFTContract : TokenRoyalty<TokenRoyaltyState>
    {
        private static readonly ByteString trueInByteString = (ByteString)new byte[] {0x01};
        private static readonly byte[] blockListPrefix = new byte[] { 0x01, 0x01 };
        public static readonly StorageMap BlocklistMap = new StorageMap(Storage.CurrentContext, blockListPrefix);
        private static readonly byte[] ownerKey = "owner".ToByteArray();
        private static readonly byte[] pausedKey = "paused".ToByteArray();
        private static readonly byte[] termsKey = "terms".ToByteArray();
        private const byte prefixTotalSupply = 0x00;
        private const byte prefixBalance = 0x01;
        private const byte prefixTokenId = 0x02;
        private const byte prefixToken = 0x03;
        private const byte prefixAccountToken = 0x04;
        public byte Decimals() => 0;

        // events
        public delegate void OnTransferDelegate(UInt160 from, UInt160 to, BigInteger amount, ByteString tokenId);

        [DisplayName("Transfer")]
        public static event OnTransferDelegate OnTransfer;
    
        // OwnershipTransferred(oldOwner, newOwner)
        public static event Action<UInt160, UInt160> OwnershipTransferred;
       
        public static event Action<UInt160, Boolean> IsUserBlocked;
        public static event Action<Boolean> IsContractPaused;

        private static bool IsOwner() => Runtime.CheckWitness(GetOwner());

        public string Symbol() => "TantalisSMBL";

        [Safe]
        public static BigInteger TotalSupply() => (BigInteger)Storage.Get(Storage.CurrentContext, new byte[] { prefixTotalSupply });

        [Safe]
        public static BigInteger BalanceOf(UInt160 owner, ByteString tokenId)
        {
            if (owner is null || !owner.IsValid)
            {
                throw new Exception("The argument 'owner' is invalid.");
            }

            var key = owner + tokenId;
            StorageMap balanceMap = new(Storage.CurrentContext, prefixBalance);
            
            return (BigInteger)balanceMap[key];
        }

        [Safe]
        public Map<string, object> Properties(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            var token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            Map<string, object> map = new();
            map["description"] = token.Description;
            map["name"] = token.Name;
            map["image"] = token.Image;
            map["creator"] = token.Creator;
            map["createdTime"] = token.CreatedTime;
            
            return map;
        }

        [Safe]
        public static Iterator Tokens()
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            
            return tokenMap.Find(FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        [Safe]
        public static Iterator TokensOf(UInt160 owner)
        {
            if (owner is null || !owner.IsValid)
            {
                throw new Exception("The argument 'owner' is invalid.");
            }

            StorageMap accountMap = new(Storage.CurrentContext, prefixAccountToken);
            
            return accountMap.Find(owner, FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        public static bool Transfer(UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            if (Paused())
            {
                throw new Exception("Contract has been paused.");
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;

            if (IsBlocked(tx.Sender))
            {
                throw new Exception("User has been blocked.");
            }

            if (to is null || !to.IsValid)
            {
                throw new Exception("The argument 'to' is invalid.");
            }

            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            var token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            var from = tx.Sender;
            
            if (!Runtime.CheckWitness(from)) 
            {
                return false;
            }

            if (from != to)
            {
                tokenMap[tokenId] = StdLib.Serialize(token);
                UpdateBalance(from, tokenId, -amount);
                UpdateBalance(to, tokenId, +amount);
            }

            PostTransfer(from, to, tokenId, amount, data);

            return true;
        }

        public virtual void SetDefaultRoyalty(UInt160 artist, int percentage, int newFeeDenominator) {
            SetDefaultTokenRoyalty(artist, percentage, newFeeDenominator);
        }

        public static ByteString CreateToken(UInt160 to, BigInteger amount, string data) 
        {
            if (Paused())
            {
                throw new Exception("Contract has been paused.");
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;

            if (IsBlocked(tx.Sender))
            {
                throw new Exception("User has been blocked.");
            }

            Map<string, string> customData = (Map<string, string>)StdLib.JsonDeserialize(data);
            var tokenId = NewTokenId();
            
            Mint(to, tokenId, amount, new TokenState
            {
                Name = customData["name"],
                Creator = tx.Sender,
                Image = customData["image"],
                Description = customData["description"],
                CreatedTime = Runtime.Time
            });
            
            return tokenId;
        }

        public static ByteString[] CreateBatchTokens(UInt160[] receivers, string[] data, int[] amount)
        {
            if (receivers.Length != data.Length && receivers.Length != amount.Length)
            {
                throw new Exception("Recievers array size is not equal data array size or amount array size.");
            }
            
            List<ByteString> tokenIds = new();

            for (int i = 0; i < data.Length; i++)
            {
                tokenIds.Add(CreateToken(receivers[i], amount[i], data[i]));
            }
            
            return tokenIds;
        }

        public static bool BatchTransfer(UInt160[] receivers, ByteString[] tokenIds, BigInteger[] amount) {
            if (receivers.Length != tokenIds.Length)
            {
                throw new Exception("Recievers array size is not equal data array size.");
            }

            for (int i = 0; i < tokenIds.Length; i++)
            {
                Transfer(receivers[i], tokenIds[i], amount[i], String.Empty);
            }

            return true;
        }

        public static UInt160 GetOwner()
        {
            return (UInt160)ContractMap.Get(ownerKey);
        }

        public static string TermsAndConditions()
        {
            return (string)ContractMap.Get(termsKey);
        }

         public static bool TransferOwnership(UInt160 newOwner)
        {
            if (!newOwner.IsValid)
            {
                throw new Exception("The new owner address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OwnershipTransferred(tx.Sender, newOwner);
            ContractMap.Put(ownerKey, newOwner);

            return true;
        }

        [DisplayName("block")]
        public static bool Block(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Put(userAddress, trueInByteString);
            IsUserBlocked(userAddress, true);
            return true;
        }

        [DisplayName("unblock")]
        public static bool UnBlock(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Delete(userAddress);
            IsUserBlocked(userAddress, false);

            return true;
        }

        [DisplayName("isblocked")]
        public static bool IsBlocked(UInt160 userAddress)
        {
            var blocked = BlocklistMap.Get((byte[])userAddress);

            return blocked == trueInByteString;
        }

        [DisplayName("pause")]
        public static bool Pause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Put(pausedKey, trueInByteString);
            IsContractPaused(true);

            return true;
        }

        [DisplayName("unpause")]
        public static bool Unpause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Delete(pausedKey);
            IsContractPaused(false);

            return true;
        }

        [DisplayName("paused")]
        public static bool Paused()
        {
            var paused = ContractMap.Get(pausedKey);

            return paused == trueInByteString;
        }

        [DisplayName("_deploy")]
        public static void Deploy(object data, bool update)
        {
            if (update)
            {
                return;
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;
            ContractMap.Put(ownerKey, tx.Sender);
            ContractMap.Put(termsKey, "Terms and Conditions of Use \nNo intellectual property rights are being transferred to the NFT buyer. More specifically, the buyer obtains none of the following rights: \n - No Copyright \n - No Commercialization Rights \n - No Derivative Rights \n - No Rights to Claim Any Royalties upon Resale of the NFT \nThe NFT buyer has only the following rights: \n - Right to Have, Hold, Admire and Display the NFT in Private, Non-Commercial Settings \n - Right to Resell the NFT Acquired, Subject to These Same Terms and Conditions");

            if (data is not null) {
                Map<string, string> customData = (Map<string, string>)data;
                if (customData["to"] is not null &&
                    customData["image"] is not null &&
                    customData["name"] is not null &&
                    customData["description"] is not null &&
                    customData["amount"] is not null) {
                    var amount = (BigInteger)(ByteString)customData["amount"];
                    var to = (UInt160)(ByteString)customData["to"];
                    
                    Mint(to, NewTokenId(), amount, new TokenState
                    {
                        Creator = tx.Sender,
                        Name = customData["name"],
                        Image = customData["image"],
                        Description = customData["description"],
                        CreatedTime = Runtime.Time
                    });
                } else {
                    throw new InvalidOperationException("Insufficient NFT creation payload data");
                }
            }
        }

        public static bool Update(ByteString nefFile, string manifest, object data = null)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }

         private static ByteString NewTokenId()
        {
            var context = Storage.CurrentContext;
            byte[] key = new byte[] { prefixTokenId };
            var id = Storage.Get(context, key);
            Storage.Put(context, key, (BigInteger)id + 1);
            ByteString data = Runtime.ExecutingScriptHash;
            if (id is not null)
            {
                data += id;
            }

            return CryptoLib.Sha256(data);
        }

        private static void Mint(UInt160 account, ByteString tokenId, BigInteger amount, TokenState token)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            tokenMap[tokenId] = StdLib.Serialize(token);
            UpdateBalance(account, tokenId, amount);
            UpdateTotalSupply(+1);
            PostTransfer(null, account, tokenId, amount, null);
        }

        private static void UpdateBalance(UInt160 owner, ByteString tokenId, BigInteger increment)
        {
            var key = owner + tokenId;
            StorageMap balanceMap = new(Storage.CurrentContext, prefixBalance);
            BigInteger balance = (BigInteger)balanceMap[key];
            balance += increment;
            
            if (balance < 0)
            {
                throw new InvalidOperationException("Balance underflow execption");
            }
            
            if (balance.IsZero)
            {
                balanceMap.Delete(owner);
            } else {
                balanceMap.Put(key, balance);
            }

            StorageMap accountMap = new(Storage.CurrentContext, prefixAccountToken);

            if (increment > 0) 
            {
                accountMap.Put(key, 0);
            } else {
                accountMap.Delete(key);
            }
        }

        private static void PostTransfer(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount,  object data)
        {
            OnTransfer(from, to, amount, tokenId);

            if (to is not null && ContractManagement.GetContract(to) is not null) {
                Contract.Call(to, "onNEP11Payment", CallFlags.All, from, 1, tokenId, data);
            }
        }

        private static void UpdateTotalSupply(BigInteger increment)
        {
            var context = Storage.CurrentContext;
            byte[] key = new byte[] { prefixTotalSupply };
            var totalSupply = (BigInteger)Storage.Get(context, key);
            totalSupply += increment;
            Storage.Put(context, key, totalSupply);
        }

    }

    public class TokenState
    {
        public string Name;
        public string Description;
        public string Image;
        public ulong CreatedTime;
        public UInt160 Creator;
    }
}
