﻿using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Numerics;

using FluentAssertions;
using Neo;
using Neo.VM;
using NeoTestHarness;
using Xunit;
using Neo.Assertions;
using Neo.BlockchainToolkit.SmartContract;
using Neo.BlockchainToolkit.Models;
using Neo.BlockchainToolkit;
using Neo.SmartContract;

namespace TantalisNFTTests
{
    [CheckpointPath("test/bin/checkpoints/contract-deployed.neoxp-checkpoint")]
    public class TantalisNFTContractTests : IClassFixture<CheckpointFixture<TantalisNFTContractTests>>
    {
        readonly CheckpointFixture fixture;
        readonly ExpressChain chain;

        public TantalisNFTContractTests(CheckpointFixture<TantalisNFTContractTests> fixture)
        {
            this.fixture = fixture;
            this.chain = fixture.FindChain("TantalisNFTTests.neo-express");
        }

        [Fact]
        public void test_symbol_and_decimals()
        {
            using var snapshot = fixture.GetSnapshot();
            var contract = snapshot.GetContract<TantalisNFTContract>();

            using var engine = new TestApplicationEngine(snapshot, ProtocolSettings.Default);
            engine.ExecuteScript<TantalisNFTContract>(c => c.symbol(), c => c.decimals());

            engine.State.Should().Be(VMState.HALT);
            engine.ResultStack.Should().HaveCount(2);
            engine.ResultStack.Peek(0).Should().BeEquivalentTo(0);
            engine.ResultStack.Peek(1).Should().BeEquivalentTo("TantalisSMBL");
        }

        [Fact]
        public void contract_owner_in_storage()
        {
            var settings = chain.GetProtocolSettings();
            var owner = chain.GetDefaultAccount("owner").ToScriptHash(settings.AddressVersion);

            using var snapshot = fixture.GetSnapshot();

            // check to make sure contract owner stored in contract storage
            var storages = snapshot.GetContractStorages<TantalisNFTContract>();
            storages.Count().Should().Be(1);
            storages.TryGetValue("MetadataOwner", out var item).Should().BeTrue();
            item!.Should().Be(owner);
        }

        [Fact]
        public void owner_is_valid()
        {
            var settings = chain.GetProtocolSettings();
            var owner = chain.GetDefaultAccount("owner").ToScriptHash(settings.AddressVersion);
            byte[] ownerKey = Encoding.ASCII.GetBytes("owner");

            using var snapshot = fixture.GetSnapshot();

            // check to make sure contract owner stored in contract storage
            var storages = snapshot.GetContractStorages<TantalisNFTContract>();
            var contractStorage = storages.StorageMap("ContractMap");
            contractStorage.TryGetValue(ownerKey, out var item).Should().BeTrue();
            item!.Should().Be(owner);
        }

        [Fact]
        public void can_change_number()
        {
            //ToDo:: make more tests
            // var settings = chain.GetProtocolSettings();
            // var alice = chain.GetDefaultAccount("alice").ToScriptHash(settings.AddressVersion);

            // using var snapshot = fixture.GetSnapshot();

            // // ExecuteScript converts the provided expression(s) into a Neo script
            // // loads them into the engine and executes it 
            // using var engine = new TestApplicationEngine(snapshot, settings, alice);

            // engine.ExecuteScript<TantalisNFTContract>(c => c.changeNumber(42));

            // engine.State.Should().Be(VMState.HALT);
            // engine.ResultStack.Should().HaveCount(1);
            // engine.ResultStack.Peek(0).Should().BeTrue();

            // // ensure that notification is triggered
            // engine.Notifications.Should().HaveCount(1);
            // engine.Notifications[0].EventName.Should().Be("NumberChanged");
            // engine.Notifications[0].State[0].Should().BeEquivalentTo(alice);
            // engine.Notifications[0].State[1].Should().BeEquivalentTo(42);

            // // ensure correct storage item was created 
            // var storages = snapshot.GetContractStorages<TantalisNFTContract>();
            // var contractStorage = storages.StorageMap("TantalisNFTContract");
            // contractStorage.TryGetValue(alice, out var item).Should().BeTrue();
            // item!.Should().Be(42);
        }
    }
}
