using System;
using System.ComponentModel;
using System.Numerics;

using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;

namespace Dvita.SmartContracts
{
    public abstract class TokenRoyalty<RoyaltyState> : SmartContract
    where RoyaltyState : TokenRoyaltyState
    {
        protected const byte Prefix_TokenRoyalty = 0x11;
        protected static readonly byte[] contractPrefix = new byte[] { 0x01, 0x02 };

        public static readonly StorageMap ContractMap = new StorageMap(Storage.CurrentContext, contractPrefix);

        private static readonly byte[] defaultArtistKey = "defaultArtist".ToByteArray();
        private static readonly byte[] defaultPercentageKey = "defaultPercentage".ToByteArray();
        private static readonly byte[] defaultfeeDenominatorKey = "feeDenominator".ToByteArray();
        public virtual int feeDenominator() => 10000;
        public delegate void OnDefaultTokenRoyaltyDelegate(UInt160 artist, int percentage, int feeDenominator);
        public static event OnDefaultTokenRoyaltyDelegate DefaultTokenRoyaltyUpdate;

        protected virtual void SetTokenRoyalty(ByteString tokenId, RoyaltyState royalty)
        {
            StorageMap royaltyMap = new(Storage.CurrentContext, Prefix_TokenRoyalty);
            royaltyMap[tokenId] = StdLib.Serialize(royalty);
        }

        public virtual Map<string, object> RoyaltiesProperties(ByteString tokenId, BigInteger price)
        {
            StorageMap royaltyMap = new(Storage.CurrentContext, Prefix_TokenRoyalty);
            
            
            Map<string, object> map = new();
            
            var defaultArtist = (UInt160)ContractMap.Get(defaultArtistKey);
            var defaultPercentage = ContractMap.Get(defaultPercentageKey) is not null ? (int)StdLib.Deserialize(ContractMap.Get(defaultPercentageKey)) : 0;
            var defaultfeeDenominator = ContractMap.Get(defaultfeeDenominatorKey) is not null ? (int)StdLib.Deserialize(ContractMap.Get(defaultfeeDenominatorKey)) : 0;

            defaultfeeDenominator = defaultfeeDenominator != 0 ? defaultfeeDenominator : 10000;
            if(royaltyMap[tokenId] is not null) 
            {
                RoyaltyState royalty = (RoyaltyState)StdLib.Deserialize(royaltyMap[tokenId]);
                map["reciever"] = royalty.Artist;
                map["royaltyAmount"] = royalty.Percentage;
            }
            else if(defaultPercentage != 0)
            {
                map["reciever"] = defaultArtist;
                map["royaltyAmount"] = defaultPercentage;
            }
            else
            {
                map["reciever"] = UInt160.Zero;
                map["royaltyAmount"] = 0;
            }

            map["royaltyAmount"] = (BigInteger)map["royaltyAmount"] * price / defaultfeeDenominator;
            
            return map;
        }

        public virtual void DeleteTokenRoyalty(ByteString tokenId)
        {
            StorageMap royaltyMap = new(Storage.CurrentContext, Prefix_TokenRoyalty);
            
            if(royaltyMap[tokenId] is not null)
            {
                royaltyMap.Delete(tokenId);
            }
        }

        protected virtual void SetDefaultTokenRoyalty(UInt160 artist, int percentage, int newFeeDenominator)
        {
            ContractMap.Put(defaultArtistKey, artist);
            ContractMap.Put(defaultPercentageKey, StdLib.Serialize(percentage));
            ContractMap.Put(defaultfeeDenominatorKey, StdLib.Serialize(newFeeDenominator));

            DefaultTokenRoyaltyUpdate(artist, percentage, newFeeDenominator);
        }
    }
}