using System;
using System.Numerics;
using System.ComponentModel;
using System.Runtime.CompilerServices;
using Neo;
using Neo.SmartContract;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;
using Neo.SmartContract.Framework.Attributes;

namespace Dvita.SmartContracts {
    [DisplayName("Dvita.SmartContracts.SocialLedger")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA social ledger")]
	[ContractPermission("*", "*")]
    public sealed class SocialLedger: SmartContract {
        //
        // Prefixes for the SC storage map for 'Handles'
        //
        private const byte Prefix_Handles = 0x01;
        public static readonly StorageMap HandlesMap = new StorageMap(Storage.CurrentContext, Prefix_Handles);

        //
        // Prefixes for the SC storage map for 'Tokens'
        //
        private const byte Prefix_Tokens = 0x02;
        public static readonly StorageMap TokensMap = new StorageMap(Storage.CurrentContext, Prefix_Tokens);

        //
        // Prefixes for the SC storage map for 'Contract'
        //
        private const byte Prefix_Contract = 0x03;
        public static readonly StorageMap ContractMap = new StorageMap(Storage.CurrentContext, Prefix_Contract);

        //
        // Pause/Unpause infra
        //
        private static readonly byte[] PausedKey = "paused".ToByteArray();
        private static readonly ByteString TrueInByteString = (ByteString)new byte[] {0x01};
        
        public static event Action<Boolean> IsContractPaused;
        public delegate void OnPausedDelegate(bool paused);

        [DisplayName("Paused")]
        public static event OnPausedDelegate OnPaused;

        //
        // Treasury account
        //
        [InitialValue("Ng1F3oJWguiuKWrdMf4qm1v8GEg3S5zMxD", ContractParameterType.Hash160)]
        private static readonly UInt160 treasury = default;

        //
        // Owner account
        //
        [InitialValue("NZJsKhsKzi9ipzjC57zU53EVMC97zqPDKG", ContractParameterType.Hash160)]
        private static readonly UInt160 owner = default;

        //
        // NEP17 Delegates
        //
        public delegate void OnTransferDelegate(UInt160 from, UInt160 to, BigInteger amount);
        
        [DisplayName("Transfer")]
        public static event OnTransferDelegate OnTransfer;

        //
        //   IsOwner()
        //   Scope: Private
        //
        private static bool IsOwner() {
            return Runtime.CheckWitness(owner);
        }

        //
        //   IsAuthorized(address)
        //   Scope: Private
        //
        private static bool IsAuthorized(UInt160 address) {
            return Runtime.CheckWitness(address);
        }

        //
        //   ValidateAddress(address)
        //   Scope: Private
        //
        private static bool ValidateAddress(UInt160 address) {
            return address.IsValid && !address.IsZero;
        }

        //
        //    PostTransfer(from, to, amount, data)
        //    Scope: Protected
        //
        protected static void PostTransfer(UInt160 from, UInt160 to, BigInteger amount, object data) {
            OnTransfer(from, to, amount);
            if (to is not null && ContractManagement.GetContract(to) is not null) {
                Contract.Call(to, "onNEP17Payment", CallFlags.All, from, amount, data);
            }
        }

        //
        //   BalanceOf(handle, token)
        //   Scope: Public
        //
        [Safe]
        [DisplayName("balanceOf")]
        public static BigInteger BalanceOf(ByteString handle, UInt160 token) {
            if (handle == string.Empty) {
                throw new Exception("Error: The parameter 'handle' should be a non-empty string.");
            }

            if (!ValidateAddress(token)) {
                throw new Exception("Error: The parameters 'token' should be a 20-byte non-zero address.");
            }

            StorageMap handleBalanceMap = new(Storage.CurrentContext, handle);
            
            if (handleBalanceMap is null) {
                return 0;
            }

            var balance = handleBalanceMap.Get(token);

            if (balance is null) {
                return 0;
            }

            return (BigInteger)balance;
        }

        //
        //   balanceOfContract()
        //   Scope: Public
        //
        [Safe]
        [DisplayName("balanceOfContract")]
        public static ByteString BalanceOfContract(UInt160 token) {
            if (!ValidateAddress(token)) {
                throw new Exception("Error: The parameters 'token' should be a 20-byte non-zero address.");
            }

            throw new NotImplementedException();
        }

        [Safe]
        [DisplayName("tokensOf")]
        public static Iterator TokensOf(UInt160 address) {
            if (!ValidateAddress(address)) {
                throw new Exception("Error: The parameters 'address' should be a 20-byte non-zero address.");
            }

            // StorageMap accountMap = new(Storage.CurrentContext, prefixAccountToken);
            //return accountMap.Find(owner, FindOptions.KeysOnly | FindOptions.RemovePrefix);

            return null;
        }

        //
        //  TransferFromHandleToAddress(@handle, address, token, amount, sender)
        //  Withdraw: @handle -> address
        //  Scope: Owner/Admin
        //
        [DisplayName("transferFromHandleToAddress")]
        public static bool TransferFromHandleToAddress(ByteString handle, UInt160 address, UInt160 token, BigInteger amount, UInt160 sender) {
            if (Paused()) {
                throw new Exception("Error: Contract has been paused.");
            }

            if (handle == string.Empty) {
                throw new Exception("Error: The parameter 'handle' should be a non-empty string.");
            }

            if (!ValidateAddress(address)) {
                throw new Exception("Error: The parameters 'address' should be a 20-byte non-zero address.");
            }

            if (!ValidateAddress(token)) {
                throw new Exception("Error: The parameters 'token' should be a 20-byte non-zero address.");
            }

            if (amount < 0) {
                throw new Exception("Error: The parameter 'amount' must be positive number.");
            }

            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            var contract = ContractManagement.GetContract(token);

            if (contract is null) {
                throw new Exception("Error: Token contract not found.");
            }

            // Get balances by token

            StorageMap handleBalanceMap = new(Storage.CurrentContext, handle);
            var handleBalanceByToken = handleBalanceMap.Get(token);
            var oldBalance = handleBalanceByToken is not null ? (BigInteger)handleBalanceByToken : (BigInteger)0;

            if (oldBalance < amount) {
                throw new Exception("Error: Insufficient balance.");
            }

            // Call NEP-17 Token SC 'transfer'

            Contract.Call(token, "transfer", CallFlags.All, treasury, address, amount, null);

            // Update balances

            var newBalance = oldBalance - amount;

            handleBalanceMap.Put(token, newBalance);

            // Update 'Tokens' Map
            TokensMap.Put(token, token);

            // Todo: Fix order
            PostTransfer(treasury, address, amount, null);

            return true;
        }

        //
        //  TransferFromHandleToHandle(@handleFrom, @handleTo, token, amount, sender)
        //  Withdraw: @handle -> address (Clearing)
        //  Scope: Owner/Admin
        //
        [DisplayName("transferFromHandleToHandle")]
        public static bool TransferFromHandleToHandle(ByteString handleFrom, ByteString handleTo, UInt160 token, BigInteger amount, UInt160 sender) {
            if (Paused()) {
                throw new Exception("Error: Contract has been paused.");
            }

            if (handleFrom == string.Empty) {
                throw new Exception("Error: The parameter 'handleFrom' should be a non-empty string.");
            }

            if (handleTo == string.Empty) {
                throw new Exception("Error: The parameter 'handleTo' should be a non-empty string.");
            }

            if (!ValidateAddress(token)) {
                throw new Exception("Error: The parameters 'token' should be a 20-byte non-zero address.");
            }

            if (amount < 0) {
                throw new Exception("Error: The parameter 'amount' must be positive number.");
            }

            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            var contract = ContractManagement.GetContract(token);

            if (contract is null) {
                throw new Exception("Error: Token contract not found.");
            }

            // Get balances by token

            StorageMap handleFromBalanceMap = new(Storage.CurrentContext, handleFrom);
            var handleFromBalanceByToken = handleFromBalanceMap.Get(token);
            var oldFromBalance = handleFromBalanceByToken is not null ? (BigInteger)handleFromBalanceByToken : (BigInteger)0;

            if (oldFromBalance < amount) {
                throw new Exception("Error: Insufficient balance.");
            }

            StorageMap handleToBalanceMap = new(Storage.CurrentContext, handleTo);
            var handleToBalanceByToken = handleToBalanceMap.Get(token);
            var oldToBalance = handleToBalanceByToken is not null ? (BigInteger)handleToBalanceByToken : (BigInteger)0;
            
            // Update balances 'from'

            var newFromBalance = oldFromBalance - amount;
            handleFromBalanceMap.Put(token, newFromBalance);

            // Update balances 'to'

            var newToBalance = oldToBalance + amount;
            handleToBalanceMap.Put(token, newToBalance);

            // Update 'Tokens' Map
            TokensMap.Put(token, token);

            return true;
        }

        //
        //  TransferFromAddressToHandle(address, @handle, token, amount)
        //  Deposit: address -> @handle
        //  Scope: Sender
        //
        [DisplayName("transferFromAddressToHandle")]
        public static bool TransferFromAddressToHandle(UInt160 address, ByteString handle, UInt160 token, BigInteger amount) {
            if (Paused()) {
                throw new Exception("Error: Contract has been paused.");
            }

            if (!ValidateAddress(address)) {
                throw new Exception("Error: The parameters 'address' should be a 20-byte non-zero address.");
            }

            if (handle == string.Empty) {
                throw new Exception("Error: The parameter 'handle' should be a non-empty string.");
            }

            if (!ValidateAddress(token)) {
                throw new Exception("Error: The parameters 'token' should be a 20-byte non-zero address.");
            }

            if (amount < 0) {
                throw new Exception("Error: The parameter 'amount' must be positive number.");
            }

            if (!IsAuthorized(address)) {
                throw new Exception("Error: No authorization.");
            }

            var contract = ContractManagement.GetContract(token);

            if (contract is null) {
                throw new Exception("Error: Token contract not found.");
            }

            // Call NEP-17 Token SC 'transfer'

            Contract.Call(token, "transfer", CallFlags.All, address, treasury, amount, null);

            // Get balances by token

            StorageMap handleBalanceMap = new(Storage.CurrentContext, handle);
            var handleBalanceByToken = handleBalanceMap.Get(token);
            var oldBalance = handleBalanceByToken is not null ? (BigInteger)handleBalanceByToken : (BigInteger)0;
            
            // Update balances

            var newBalance = oldBalance + amount;

            handleBalanceMap.Put(token, newBalance);

            // Update 'Tokens' Map
            TokensMap.Put(token, token);

            // Todo: Fix order
            PostTransfer(address, treasury, amount, null);

            return true;
        }

        //
        //  ClaimAllToAddress(@handle, address)
        //  Claims all tokens & amounts: @handle -> address
        //  Scope: Admin/Owner
        //
        [DisplayName("claimAllToAddress")]
        public static bool ClaimAllToAddress(ByteString handle, UInt160 address, UInt160 sender) {
            if (Paused()) {
                throw new Exception("Error: Contract has been paused.");
            }

            if (handle == string.Empty) {
                throw new Exception("Error: The parameter 'handle' should be a non-empty string.");
            }

            if (!ValidateAddress(address)) {
                throw new Exception("Error: The parameters 'address' should be a 20-byte non-zero address.");
            }

            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            // 1. Get user's balances map
            StorageMap handleBalanceMap = new(Storage.CurrentContext, handle);

            // 2. Find all user's tokens aka keys of map above and call 'Transfer' on it
            var retVal = false;
            var handleBalanceMapIterator = handleBalanceMap.Find(FindOptions.RemovePrefix | FindOptions.KeysOnly);
            while (handleBalanceMapIterator.Next()) {
                var tokenHash = (UInt160)handleBalanceMapIterator.Value;
                var tokenValue = handleBalanceMap.Get(tokenHash);
                var tokenValueNormalized = tokenValue is not null ? (BigInteger)tokenValue : (BigInteger)0;

                var contract = ContractManagement.GetContract(tokenHash);

                if (contract is not null && !tokenValueNormalized.IsZero) {
                    // Call NEP-17 Token SC 'transfer'
                    Contract.Call(tokenHash, "transfer", CallFlags.All, treasury, address, tokenValueNormalized, null);

                    // Reset handle balance since its amount was withdrawn
                    handleBalanceMap.Put(tokenHash, 0);

                    // Todo: Fix order
                    PostTransfer(treasury, address, tokenValueNormalized, null);
                    
                    retVal = true;
                }
            }

            return retVal;
        }

        //
        //   GetTrialBalanceReport()
        //   Scope: Owner/Admin
        //
        [Safe]
        [DisplayName("getTrialBalanceReport")]
        public static ByteString GetTrialBalanceReport(UInt160 sender) {
            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            throw new NotImplementedException();
        }

        //
        //   GetAccumulatedBalanceReport()
        //   Scope: Owner/Admin
        //
        [Safe]
        [DisplayName("getAccumulatedBalanceReport")]
        public static ByteString GetAccumulatedBalanceReport(UInt160 sender) {
            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            throw new NotImplementedException();
        }

        //
        //  DumpLedger(sender)
        //  Scope: Admin/Owner
        //
        [Safe]
        [DisplayName("dumpLedger")]
        public static ByteString DumpLedger(UInt160 sender) {
            if (!ValidateAddress(sender)) {
                throw new Exception("Error: The parameters 'sender' should be a 20-byte non-zero address.");
            }

            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            throw new Exception("Error: Not implemented yet.");
            
            return "".ToByteArray().ToByteString();
        }

        //
        //  Pause()
        //  Scope: Admin/Owner
        //
        [DisplayName("pause")]
        public static bool Pause() {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            ContractMap.Put(PausedKey, TrueInByteString);
            IsContractPaused(true);
            OnPaused(true);
            return true;
        }

        //
        //  Unpause()
        //  Scope: Admin/Owner
        //
        [DisplayName("unpause")]
        public static bool Unpause() {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            ContractMap.Delete(PausedKey);
            IsContractPaused(false);
            OnPaused(false);
            return true;
        }

        //
        //  Paused()
        //  Scope: Public
        //
        [DisplayName("paused")]
        public static bool Paused() {
            var paused = ContractMap.Get(PausedKey);
            return paused == TrueInByteString;
        }

        //
        //  Deploy(data, update)
        //  Scope: System
        //
        [DisplayName("_deploy")]
        public static void Deploy(object data, bool update) {
            if (update) {
                return;
            }
        }

        //
        //  Update(nefFile, manifest, data)
        //  Scope: Admin/Owner
        //
        public static bool Update(ByteString nefFile, string manifest, object data = null) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }
    }
}
