using System;
using System.Numerics;
using System.ComponentModel;
using System.Runtime.CompilerServices;
using Neo;
using Neo.SmartContract;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;
using Neo.SmartContract.Framework.Attributes;

namespace Dvita.SmartContracts {
    [DisplayName("Dvita.SmartContracts.CnameResolver")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA canonical name resolver")]
	[ContractPermission("*", "*")]
    public sealed class CnameResolver: SmartContract {
        //
        // Prefixes for the SC storage map for 'NamesMap'
        //
        private const byte Prefix_Names = 0x01;
        public static readonly StorageMap NamesMap = new StorageMap(Storage.CurrentContext, Prefix_Names);

        //
        // Owner account
        //
        [InitialValue("NZJsKhsKzi9ipzjC57zU53EVMC97zqPDKG", ContractParameterType.Hash160)]
        private static readonly UInt160 owner = default;

        //
        //   IsOwner()
        //   Scope: Private
        //
        private static bool IsOwner() {
            return Runtime.CheckWitness(owner);
        }

        //
        //  Register(cname, address)
        //  Scope: Owner/Admin
        //
        [DisplayName("register")]
        public static bool Register(ByteString cname, UInt160 address) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: No Authorization");
            }

            var existent = (UInt160)NamesMap.Get(cname);
            if (existent != null) {
                throw new InvalidOperationException("Error: Name already exists.");
            }

            NamesMap.Put(cname, address);

            return true;
        }

        //
        //  RegisterBulk(cnames[], addresses[])
        //  Scope: Owner/Admin
        //
        [DisplayName("registerBulk")]
        public static bool Register(ByteString[] cnames, UInt160[] addresses) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: No Authorization");
            }

            if (cnames.Length != addresses.Length) {
                 throw new InvalidOperationException("Given 'cnames' array size is not equal to 'addresses' array size.");
            }

            for (int i = 0; i < cnames.Length; i++) {
                Register(cnames[i], addresses[i]);
            }

            return true;
        }

        //
        //  Unregister(cname)
        //  Scope: Owner/Admin
        //
        [DisplayName("unregister")]
        public static bool Unregister(ByteString cname) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: No Authorization");
            }

            var existent = NamesMap.Get(cname);

            if (existent == null) {
                throw new InvalidOperationException("Error: Name not found.");
            }

            NamesMap.Delete(cname);

            return true;
        }

        //
        //   Resolve(cname)
        //   Scope: Public
        //
        [Safe]
        [DisplayName("resolve")]
        public static UInt160 Resolve(ByteString cname) {
            var existent = (UInt160)NamesMap.Get(cname);

            if (existent != null) {
                return existent;
            }

            throw new InvalidOperationException("Error: Name not found.");
        }

        //
        //   ReverseResolve(address)
        //   Scope: Owner/Admin
        //
        [Safe]
        [DisplayName("reverseResolve")]
        public static ByteString ReverseResolve(UInt160 address) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: No Authorization");
            }

            var namesMapIterator = NamesMap.Find(FindOptions.RemovePrefix | FindOptions.KeysOnly);
            while (namesMapIterator.Next()) {
                var cname = (ByteString)namesMapIterator.Value;
                var addressByCname = (UInt160)NamesMap.Get(cname);
                if (address == addressByCname) {
                    return cname;
                }
            }

            throw new InvalidOperationException("Error: Name not found.");
        }

        //
        //  Dump()
        //  Scope: Owner/Admin
        //
        [Safe]
        [DisplayName("dump")]
        public static ByteString Dump() {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: No Authorization");
            }

            var dumpByteArray = "Registry dump:\n".ToByteArray();

            var namesMapIterator = NamesMap.Find(FindOptions.RemovePrefix | FindOptions.KeysOnly);
            while (namesMapIterator.Next()) {
                var cname = (ByteString)namesMapIterator.Value;
                var address = (UInt160)NamesMap.Get(cname);
                dumpByteArray = dumpByteArray
                    .Concat(cname)
                    .Concat(" - ".ToByteArray())
                    .Concat(StdLib.Base64Encode((ByteString)address).ToByteArray())
                    .Concat(";\n".ToByteArray());
            }
            
            return dumpByteArray.ToByteString();
        }

        //
        //  Deploy(data, update)
        //  Scope: System
        //
        [DisplayName("_deploy")]
        public static void Deploy(object data, bool update) {
            if (update) {
                return;
            }
        }

        //
        //  Update(nefFile, manifest, data)
        //  Scope: Admin/Owner
        //
        public static bool Update(ByteString nefFile, string manifest, object data = null) {
            if (!IsOwner()) {
                throw new InvalidOperationException("Error: Caller is not the owner.");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }
    }
}
