﻿using System;
using System.ComponentModel;
using System.Numerics;
using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;
using Neo.Cryptography.ECC;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.Marketplace")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA NFT marketplace")]
	[ContractPermission("*", "*")]

    public class Marketplace : SmartContract
    {
        private static readonly ByteString trueInByteString = (ByteString)new byte[] {0x01};
        private static readonly byte[] blockListPrefix = new byte[] { 0x01, 0x01 };
        private static readonly byte[] contractPrefix = new byte[] { 0x01, 0x02 };
        public static readonly StorageMap ContractMap = new StorageMap(Storage.CurrentContext, contractPrefix);
        public static readonly StorageMap BlocklistMap = new StorageMap(Storage.CurrentContext, blockListPrefix);
        private static readonly byte[] ownerKey = "owner".ToByteArray();
        private static readonly byte[] executorKey = "executorKey".ToByteArray();
        private static readonly byte[] validatorKey = "validatorKey".ToByteArray();
        private static readonly byte[] pausedKey = "paused".ToByteArray();

        // events
        [DisplayName("ExportedNFT")]
        public static event Action<UInt160, UInt160, ByteString> OnExportedNFT;
        // OwnershipTransferred(oldOwner, newOwner)
        [DisplayName("OwnershipTransferred")]
        public static event Action<UInt160, UInt160> OnOwnershipTransferred;
        [DisplayName("ExecutorTransferred")]
        public static event Action<UInt160, UInt160> OnExecutorTransferred;
        [DisplayName("ValidatorTransferred")]
        public static event Action<UInt160, ECPoint> OnValidatorTransferred;
        public static event Action<UInt160, bool> isUserBlocked;
        public static event Action<bool> isContractPaused;
        
        private static bool IsOwner() => Runtime.CheckWitness(GetOwner());
        private static bool IsExecutor() => Runtime.CheckWitness(GetExecutor());
        private static bool IsValidator(ECPoint pubkey) => (GetValidator() == pubkey);

        public static void ExportNFT(string message, string signedMessage, ECPoint pubkey)
        {
            if (Paused())
            {
                throw new Exception("Contract has been paused.");
            }

            if (!IsExecutor())
            {
                throw new Exception("Caller is not the executor.");
            }

            if (!IsValidator(pubkey))
            {
                throw new Exception("Not Validator's pubkey.");
            }

            if(!CryptoLib.VerifyWithECDsa(message, pubkey, signedMessage, NamedCurve.secp256r1)) 
            {
                throw new Exception("Wrong signature.");
            }

            Map<string, string> customData = (Map<string, string>)StdLib.JsonDeserialize(message);

            if (customData["to"] is null
            || customData["nft"] is null
            || customData["tokenid"] is null)
            {
                throw new Exception("There are not enough data in the message");
            }

            var to = (UInt160)(ByteString) customData["to"];
            var nft = (UInt160)(ByteString) customData["nft"];
            var tokenId = (ByteString) customData["tokenid"];

            Contract.Call(nft, "transfer", CallFlags.All, new object[] { to, tokenId, null });
            OnExportedNFT(to, nft, tokenId);
        }

        public static void OnNEP11Payment(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            return;
        }

        public static UInt160 GetOwner()
        {
            return (UInt160)ContractMap.Get(ownerKey);
        }

        public static UInt160 GetExecutor()
        {
            return (UInt160)ContractMap.Get(executorKey);
        }

        public static ECPoint GetValidator()
        {
            return (ECPoint)ContractMap.Get(validatorKey);
        }

        public static bool TransferOwnership(UInt160 newOwner)
        {
            if (!newOwner.IsValid)
            {
                throw new Exception("The new owner address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnOwnershipTransferred(tx.Sender, newOwner);
            ContractMap.Put(ownerKey, newOwner);

            return true;
        }

        public static bool TransferExecutor(UInt160 newExecutor)
        {
            if (!newExecutor.IsValid)
            {
                throw new Exception("The new executor address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnExecutorTransferred(tx.Sender, newExecutor);
            ContractMap.Put(executorKey, newExecutor);

            return true;
        }

        public static bool TransferValidator(ECPoint newValidator)
        {
            if (!newValidator.IsValid)
            {
                throw new Exception("The new validator address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnValidatorTransferred(tx.Sender, newValidator);
            ContractMap.Put(validatorKey, newValidator);

            return true;
        }

        [DisplayName("block")]
        public static bool Block(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Put(userAddress, trueInByteString);
            isUserBlocked(userAddress, true);
            return true;
        }

        [DisplayName("unblock")]
        public static bool UnBlock(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Delete(userAddress);
            isUserBlocked(userAddress, false);

            return true;
        }

        [DisplayName("isblocked")]
        public static bool IsBlocked(UInt160 userAddress)
        {
            var blocked = BlocklistMap.Get(userAddress);

            return blocked == trueInByteString;
        }

        [DisplayName("pause")]
        public static bool Pause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Put(pausedKey, trueInByteString);
            isContractPaused(true);

            return true;
        }

        [DisplayName("unpause")]
        public static bool Unpause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Delete(pausedKey);
            isContractPaused(false);

            return true;
        }

        [DisplayName("paused")]
        public static bool Paused()
        {
            var paused = ContractMap.Get(pausedKey);

            return paused == trueInByteString;
        }

        [DisplayName("_deploy")]
        public static void Deploy(object data, bool update)
        {
            if (update)
            {
                return;
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;
            ContractMap.Put(ownerKey, tx.Sender);
        }

        public static bool Update(ByteString nefFile, string manifest, object data = null)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }
    }
}
