using System;
using System.ComponentModel;
using System.Numerics;

using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.Dep721Token")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA Dep721 Standart NFT")]
	[ContractPermission("*", "onDep721Payment")]
    public abstract class Dep721Token<TokenState, RoyaltyState> : TokenRoyalty<RoyaltyState>
        where TokenState : Dep721TokenState
        where RoyaltyState : TokenRoyaltyState
    {
        public delegate void OnTransferDelegate(UInt160 from, UInt160 to, BigInteger amount, ByteString tokenId);

        [DisplayName("Transfer")]
        public static event OnTransferDelegate OnTransfer;

        public delegate void OnApproveDelegate(UInt160 owner, UInt160 to, BigInteger amount, ByteString tokenId);

        [DisplayName("Approval")]
        public static event OnApproveDelegate OnApprove;

        public delegate void OnApproveForAllDelegate(UInt160 owner, UInt160 to, BigInteger amount, bool approved);

        [DisplayName("ApprovalForAll")]
        public static event OnApproveForAllDelegate OnApproveForAll;

        protected const byte Prefix_TokenId = 0x02;
        protected const byte Prefix_Token = 0x03;
        protected const byte Prefix_AccountToken = 0x04;
        protected const byte Prefix_TokenApprovals = 0x05;
        protected const byte Prefix_OperatorApprovals = 0x06;

        public class TokenApprovals
        {
            public Map<UInt160, bool> isApproved;
        }
        
        [Safe]
        public sealed override byte Decimals() => 0;

        [Safe]
        public virtual UInt160 OwnerOf(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            
            return token.Owner;
        }

        [Safe]
        public virtual Map<string, object> Properties(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            Map<string, object> map = new();
            map["name"] = token.Name;
            
            return map;
        }

        [Safe]
        public virtual Iterator Tokens()
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            
            return tokenMap.Find(FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        [Safe]
        public virtual Iterator TokensOf(UInt160 owner)
        {
            if (owner is null || !owner.IsValid)
            {
                throw new Exception("The argument 'owner' is invalid");
            }

            StorageMap accountMap = new(Storage.CurrentContext, Prefix_AccountToken);
            
            return accountMap.Find(owner, FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        public virtual bool Transfer(UInt160 to, ByteString tokenId, object data)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            UInt160 from = token.Owner;

            if (!Runtime.CheckWitness(from)) 
            {
                return false;
            }

            return InTransfer(from, to, tokenId, data);
        }

        public virtual void Approve(UInt160 to, ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            UInt160 from = token.Owner;

            if (!Runtime.CheckWitness(from)) 
            {
                throw new Exception("The Sender is not the 'owner'");
            }

            InApprove(to, tokenId);
        }

        public virtual bool SetApprovalForAll(UInt160 caller, bool approved)
        {
            Transaction tx = (Transaction)Runtime.ScriptContainer;
            UInt160 owner = tx.Sender;

            return InApprovalForAll(owner, caller, approved);
        }

        public virtual bool IsApprovedForAll(UInt160 owner, UInt160 caller)
        {
            if(owner is null || caller is null)
            {
                return false;
            }
            
            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, Prefix_OperatorApprovals);
            var key = Helper.Concat((byte[])owner, (byte[])caller);
            
            if(tokenApprovalsMap[key] is not null && (bool)StdLib.Deserialize(tokenApprovalsMap[key]))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public virtual bool TokenExists(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            return token.Owner != UInt160.Zero; // && token.Name != null;
        }

        public virtual UInt160 GetApproved(ByteString tokenId)
        {
            if (!TokenExists(tokenId)) 
            {
                throw new Exception("The token is not exist");
            }

            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, Prefix_TokenApprovals);
            
            if(tokenApprovalsMap[tokenId] is not null)
            {
                return (UInt160)tokenApprovalsMap[tokenId];
            }

            return UInt160.Zero;
        }

        public virtual bool TransferFrom(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            Transaction tx = (Transaction)Runtime.ScriptContainer;
            UInt160 caller = tx.Sender;

            if (GetApproved(tokenId) != caller && !IsApprovedForAll(from, caller) && caller != from) 
            {
                throw new Exception("The Sender is not approved to transfer");
            }

            return InTransfer(from, to, tokenId, data);
        }

        protected virtual bool InTransfer(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            PreTransfer(from, to, tokenId, null);
            
            if (to is null || !to.IsValid)
            {
                throw new Exception("The argument 'to' is invalid");
            }

            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token;

            if(tokenMap[tokenId] is not null)
            {
                token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            }
            else 
            {
                return false;
            }

            if (from != to)
            {
                token.Owner = to;
                tokenMap[tokenId] = StdLib.Serialize(token);
                UpdateBalance(from, tokenId, -1);
                UpdateBalance(to, tokenId, +1);
            }

            InApprove(UInt160.Zero, tokenId);
            PostTransfer(from, to, tokenId, data);

            return true;
        }

        protected virtual void InApprove(UInt160 to, ByteString tokenId)
        {
            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, Prefix_TokenApprovals);
            
            tokenApprovalsMap.Put(tokenId, to);
            OnApprove(OwnerOf(tokenId), to, 1, tokenId);
        }

        protected virtual bool InApprovalForAll(UInt160 from, UInt160 to, bool approved)
        {
            if (from == to) 
            {
                throw new Exception("Cannot approve to the owner");
            }

            if(from is null || to is null)
            {
                return false;
            }
            
            var key = Helper.Concat((byte[])from, (byte[])to);
            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, Prefix_OperatorApprovals);
            tokenApprovalsMap.Put(key, StdLib.Serialize(approved));
            OnApproveForAll(from, to, 1, approved);
            
            return true;
        }

        protected virtual ByteString NewTokenId()
        {
            return NewTokenId(Runtime.ExecutingScriptHash);
        }

        protected virtual ByteString NewTokenId(ByteString salt)
        {
            StorageContext context = Storage.CurrentContext;
            byte[] key = new byte[] { Prefix_TokenId };
            ByteString id = Storage.Get(context, key);
            Storage.Put(context, key, (BigInteger)id + 1);
            
            if (id is not null)
            {
                salt += id;
            }

            return CryptoLib.Sha256(salt);
        }

        protected virtual void Mint(ByteString tokenId, TokenState token)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            tokenMap[tokenId] = StdLib.Serialize(token);
            PreTransfer(null, token.Owner, tokenId, null);
            UpdateBalance(token.Owner, tokenId, +1);
            UpdateTotalSupply(+1);
            PostTransfer(null, token.Owner, tokenId, null);
        }

        protected virtual void Burn(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, Prefix_Token);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            PreTransfer(token.Owner, null, tokenId, null);
            tokenMap.Delete(tokenId);
            UpdateBalance(token.Owner, tokenId, -1);
            UpdateTotalSupply(-1);
            PostTransfer(token.Owner, null, tokenId, null);
        }

        protected virtual void UpdateBalance(UInt160 owner, ByteString tokenId, int increment)
        {
            UpdateBalance(owner, increment);
            StorageMap accountMap = new(Storage.CurrentContext, Prefix_AccountToken);
            ByteString key = owner + tokenId;
            
            if (increment > 0)
            {
                accountMap.Put(key, 0);
            } 
            else
            {
                accountMap.Delete(key);
            }
        }

        protected virtual void PreTransfer(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            //
        }

        protected virtual void PostTransfer(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            OnTransfer(from, to, 1, tokenId);

            if (to is not null && ContractManagement.GetContract(to) is not null)
                Contract.Call(to, "onDEP721Payment", CallFlags.All, from, 1, tokenId, data);
        }
    }
}