﻿using System;
using System.ComponentModel;
using System.Numerics;
using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;
using Neo.Cryptography.ECC;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.DepMarketplace")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA Dep721NFT marketplace")]
    [ContractPermission("*", "*")]
    public class Dep721MarketplaceContract : SmartContract
    {
        private static readonly ByteString TrueInByteString = (ByteString)new byte[] {0x01};
        private static readonly byte[] BlockListPrefix = new byte[] { 0x01, 0x01 };
        private static readonly byte[] ContractPrefix = new byte[] { 0x01, 0x02 };
        public static readonly StorageMap ContractMap = new StorageMap(Storage.CurrentContext, ContractPrefix);
        public static readonly StorageMap BlocklistMap = new StorageMap(Storage.CurrentContext, BlockListPrefix);
        private static readonly byte[] OwnerKey = "owner".ToByteArray();
        private static readonly byte[] ExecutorKey = "executor".ToByteArray();
        private static readonly byte[] ValidatorKey = "validator".ToByteArray();
        private static readonly byte[] PausedKey = "paused".ToByteArray();

        [DisplayName("BuyNFT")]
        public static event Action<UInt160, UInt160, UInt160, ByteString> OnBuyNFT;
        [DisplayName("ExportedNFT")]
        public static event Action<UInt160, UInt160, ByteString> OnExportedNFT;
        [DisplayName("OwnershipTransferred")]
        public static event Action<UInt160, UInt160> OnOwnershipTransferred;
        [DisplayName("ExecutorTransferred")]
        public static event Action<UInt160, UInt160> OnExecutorTransferred;
        [DisplayName("ValidatorTransferred")]
        public static event Action<UInt160, ECPoint> OnValidatorTransferred;
        public static event Action<UInt160, bool> IsUserBlocked;
        public static event Action<bool> IsContractPaused;
        
        [Safe]
        private static bool IsOwner() => Runtime.CheckWitness(GetOwner());

        [Safe]
        private static bool IsExecutor() => Runtime.CheckWitness(GetExecutor());

        [Safe]
        private static bool IsValidator(ECPoint pubkey) => (GetValidator() == pubkey);

        public static void BuyNFT(string message, string signedMessage, ECPoint pubkey)
        {
            ValidateData(message, signedMessage, pubkey);

            Map<string, string> customData = (Map<string, string>)StdLib.JsonDeserialize(message);

            if (customData["from"] is null
            || customData["to"] is null
            || customData["nft"] is null
            || customData["tokenid"] is null)
            {
                throw new Exception("There are not enough data in the message");
            }

            // ToDo:: are there any issues with type casting
            var from = (UInt160)(ByteString) customData["from"];
            var to = (UInt160)(ByteString) customData["to"];
            var nft = (UInt160)(ByteString) customData["nft"];
            var tokenId = (ByteString) customData["tokenid"];

            Contract.Call(nft, "transferFrom", CallFlags.All, new object[] { from, to, tokenId, null });
            OnBuyNFT(from, to, nft, tokenId);
        }

        public static void ExportNFT(string message, string signedMessage, ECPoint pubkey)
        {
            ValidateData(message, signedMessage, pubkey);

            Map<string, string> customData = (Map<string, string>)StdLib.JsonDeserialize(message);

            if (customData["to"] is null
            || customData["nft"] is null
            || customData["tokenid"] is null)
            {
                throw new Exception("There are not enough data in the message");
            }

            // ToDo:: are there any issues with type casting
            var to = (UInt160)(ByteString) customData["to"];
            var nft = (UInt160)(ByteString) customData["nft"];
            var tokenId = (ByteString) customData["tokenid"];

            Contract.Call(nft, "transfer", CallFlags.All, new object[] { to, tokenId, null });
            OnExportedNFT(to, nft, tokenId);
        }

        public static void OnNEP11Payment(UInt160 from, UInt160 to, ByteString tokenId, object data)
        {
            return;
        }

        public static UInt160 GetOwner()
        {
            return (UInt160)ContractMap.Get(OwnerKey);
        }

        public static UInt160 GetExecutor()
        {
            return (UInt160)ContractMap.Get(ExecutorKey);
        }

        public static ECPoint GetValidator()
        {
            return (ECPoint)ContractMap.Get(ValidatorKey);
        }

        public static bool TransferOwnership(UInt160 newOwner)
        {
            if (!newOwner.IsValid)
            {
                throw new Exception("The new owner address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnOwnershipTransferred(tx.Sender, newOwner);
            ContractMap.Put(OwnerKey, newOwner);

            return true;
        }

        public static bool TransferExecutor(UInt160 newExecutor)
        {
            if (!newExecutor.IsValid)
            {
                throw new Exception("The new executor address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnExecutorTransferred(tx.Sender, newExecutor);
            ContractMap.Put(ExecutorKey, newExecutor);

            return true;
        }

        public static bool TransferValidator(ECPoint newValidator)
        {
            if (!newValidator.IsValid)
            {
                throw new Exception("The new validator address is invalid.");
            }

            if (!IsOwner())
            {
                throw new Exception("Caller is not the owner.");
            }

            Transaction tx = (Transaction)Runtime.ScriptContainer;
            OnValidatorTransferred(tx.Sender, newValidator);
            ContractMap.Put(ValidatorKey, newValidator);

            return true;
        }

        [DisplayName("block")]
        public static bool Block(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Put(userAddress, TrueInByteString);
            IsUserBlocked(userAddress, true);
            return true;
        }

        [DisplayName("unblock")]
        public static bool UnBlock(UInt160 userAddress)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            BlocklistMap.Delete(userAddress);
            IsUserBlocked(userAddress, false);

            return true;
        }

        [Safe]
        [DisplayName("isblocked")]
        public static bool IsBlocked(UInt160 userAddress)
        {
            var blocked = BlocklistMap.Get(userAddress);

            return blocked == TrueInByteString;
        }

        [DisplayName("pause")]
        public static bool Pause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Put(PausedKey, TrueInByteString);
            IsContractPaused(true);

            return true;
        }

        [DisplayName("unpause")]
        public static bool Unpause()
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractMap.Delete(PausedKey);
            IsContractPaused(false);

            return true;
        }

        [Safe]
        [DisplayName("paused")]
        public static bool Paused()
        {
            var paused = ContractMap.Get(PausedKey);

            return paused == TrueInByteString;
        }

        [DisplayName("_deploy")]
        public static void Deploy(object data, bool update)
        {
            if (update)
            {
                return;
            }
            
            var tx = (Transaction)Runtime.ScriptContainer;
            ContractMap.Put(OwnerKey, tx.Sender);
        }

        public static bool Update(ByteString nefFile, string manifest, object data = null)
        {
            if (!IsOwner())
            {
                throw new InvalidOperationException("Caller is not the owner.");
            }

            ContractManagement.Update(nefFile, manifest, data);

            return true;
        }
        
        protected static void ValidateData(string message, string signedMessage, ECPoint pubkey)
        {
            if (Paused())
            {
                throw new Exception("Contract has been paused.");
            }

            if (!IsExecutor())
            {
                throw new Exception("Caller is not the executor.");
            }

            if (!IsValidator(pubkey))
            {
                throw new Exception("Not Validator's pubkey.");
            }

            if(!CryptoLib.VerifyWithECDsa(message, pubkey, signedMessage, NamedCurve.secp256r1)) 
            {
                throw new Exception("Wrong signature.");
            }
        }
    }
}
