using System;
using System.ComponentModel;
using System.Numerics;

using Neo;
using Neo.SmartContract.Framework;
using Neo.SmartContract.Framework.Native;
using Neo.SmartContract.Framework.Services;

namespace Dvita.SmartContracts
{
    [DisplayName("Dvita.SmartContracts.Dep1155Token")]
    [ManifestExtra("Author", "DVITA Team")]
    [ManifestExtra("Email", "info@dvita.com")]
    [ManifestExtra("Description", "DVITA Dep1155 Standart NFT")]
	[ContractPermission("*", "onDep1155Payment")]
    public abstract class Dep1155Token<TokenState, RoyaltyState> : TokenRoyalty<RoyaltyState>
        where TokenState : Dep1155TokenState
        where RoyaltyState : TokenRoyaltyState
    {
        public delegate void OnTransferDelegate(UInt160 from, UInt160 to, BigInteger amount, ByteString tokenId);

        [DisplayName("Transfer")]
        public static event OnTransferDelegate OnTransfer;

        public delegate void OnApproveDelegate(UInt160 owner, UInt160 to, BigInteger amount, ByteString tokenId);

        public delegate void OnApproveForAllDelegate(UInt160 owner, UInt160 to, BigInteger amount, bool approved);

        [DisplayName("ApprovalForAll")]
        public static event OnApproveForAllDelegate OnApproveForAll;

        protected const byte prefixTokenId = 0x02;
        protected const byte prefixToken = 0x03;
        protected const byte prefixAccountToken = 0x04;
        protected const byte prefixTokenApprovals = 0x05;
        protected const byte prefixOperatorApprovals = 0x06;

        public class TokenApprovals
        {
            public Map<UInt160, bool> isApproved;
        }
        
        [Safe]
        public sealed override byte Decimals() => 0;

        [Safe]
        public static BigInteger BalanceOf(UInt160 owner, ByteString tokenId)
        {
            if (owner is null || !owner.IsValid)
            {
                throw new Exception("The argument 'owner' is invalid.");
            }

            var key = owner + tokenId;
            StorageMap balanceMap = new(Storage.CurrentContext, prefixBalance);
            
            return (BigInteger)balanceMap[key];
        }

        [Safe]
        public virtual Map<string, object> Properties(ByteString tokenId)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            Map<string, object> map = new();
            map["name"] = token.Name;
            
            return map;
        }

        [Safe]
        public virtual Iterator Tokens()
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            
            return tokenMap.Find(FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        [Safe]
        public virtual Iterator TokensOf(UInt160 owner)
        {
            if (owner is null || !owner.IsValid)
            {
                throw new Exception("The argument 'owner' is invalid");
            }

            StorageMap accountMap = new(Storage.CurrentContext, prefixAccountToken);
            
            return accountMap.Find(owner, FindOptions.KeysOnly | FindOptions.RemovePrefix);
        }

        public virtual bool Transfer(UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            var tx = (Transaction)Runtime.ScriptContainer;
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            TokenState token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            var from = tx.Sender;

            if (!Runtime.CheckWitness(from)) 
            {
                return false;
            }

            return InTransfer(from, to, tokenId, amount, data);
        }

        public virtual bool BatchTransfer(UInt160[] receivers, ByteString[] tokenIds, BigInteger[] amount) {
            if (receivers.Length != tokenIds.Length && receivers.Length != amount.Length)
            {
                throw new Exception("Recievers array size is not equal data array size or amount array size");
            }

            for (int i = 0; i < tokenIds.Length; i++)
            {
                Transfer(receivers[i], tokenIds[i], amount[i], String.Empty);
            }

            return true;
        }

        public virtual bool SetApprovalForAll(UInt160 caller, bool approved)
        {
            Transaction tx = (Transaction)Runtime.ScriptContainer;
            UInt160 owner = tx.Sender;

            return InApprovalForAll(owner, caller, approved);
        }

        public virtual bool IsApprovedForAll(UInt160 owner, UInt160 caller)
        {
            if(owner is null || caller is null)
            {
                return false;
            }
            
            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, prefixOperatorApprovals);
            var key = Helper.Concat((byte[])owner, (byte[])caller);
            
            if(tokenApprovalsMap[key] is not null && (bool)StdLib.Deserialize(tokenApprovalsMap[key]))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public virtual bool TransferFrom(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            Transaction tx = (Transaction)Runtime.ScriptContainer;
            UInt160 caller = tx.Sender;

            if (!IsApprovedForAll(from, caller) && caller != from) 
            {
                throw new Exception("The Sender is not approved to transfer");
            }

            return InTransfer(from, to, tokenId, amount, data);
        }

        protected virtual bool InTransfer(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            PreTransfer(from, to, tokenId, amount, null);
            
            if (to is null || !to.IsValid)
            {
                throw new Exception("The argument 'to' is invalid");
            }

            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            TokenState token;

            if(tokenMap[tokenId] is not null)
            {
                token = (TokenState)StdLib.Deserialize(tokenMap[tokenId]);
            }
            else 
            {
                return false;
            }

            if (from != to)
            {
                tokenMap[tokenId] = StdLib.Serialize(token);
                UpdateBalance(from, tokenId, -amount);
                UpdateBalance(to, tokenId, +amount);
            }

            PostTransfer(from, to, tokenId, amount, data);

            return true;
        }

        protected virtual bool InApprovalForAll(UInt160 from, UInt160 to, bool approved)
        {
            if (from == to) 
            {
                throw new Exception("Cannot approve to the owner");
            }

            if(from is null || to is null)
            {
                return false;
            }
            
            var key = Helper.Concat((byte[])from, (byte[])to);
            StorageMap tokenApprovalsMap = new(Storage.CurrentContext, prefixOperatorApprovals);
            tokenApprovalsMap.Put(key, StdLib.Serialize(approved));
            OnApproveForAll(from, to, 1, approved);
            
            return true;
        }

        protected virtual ByteString NewTokenId()
        {
            return NewTokenId(Runtime.ExecutingScriptHash);
        }

        protected virtual ByteString NewTokenId(ByteString salt)
        {
            StorageContext context = Storage.CurrentContext;
            byte[] key = new byte[] { prefixTokenId };
            ByteString id = Storage.Get(context, key);
            Storage.Put(context, key, (BigInteger)id + 1);
            
            if (id is not null)
            {
                salt += id;
            }

            return CryptoLib.Sha256(salt);
        }

        protected virtual void Mint(UInt160 account, ByteString tokenId, BigInteger amount, TokenState token)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            tokenMap[tokenId] = StdLib.Serialize(token);
            PreTransfer(null, account, tokenId, amount, null);
            UpdateBalance(account, tokenId, amount);
            UpdateTotalSupply(+1);
            PostTransfer(null, account, tokenId, amount, null);
        }

        protected virtual void Burn(UInt160 account, ByteString tokenId, BigInteger amount, TokenState token)
        {
            StorageMap tokenMap = new(Storage.CurrentContext, prefixToken);
            PreTransfer(account, null, tokenId, amount, null);
            tokenMap.Delete(tokenId);
            UpdateBalance(account, tokenId, -amount);
            UpdateTotalSupply(-1);
            PostTransfer(account, null, tokenId, amount, null);
        }

        protected virtual void PreTransfer(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            //
        }

        protected virtual void PostTransfer(UInt160 from, UInt160 to, ByteString tokenId, BigInteger amount, object data)
        {
            OnTransfer(from, to, amount, tokenId);

            if (to is not null && ContractManagement.GetContract(to) is not null)
                Contract.Call(to, "onDep1155Payment", CallFlags.All, from, 1, tokenId, data);
        }

        protected virtual void UpdateBalance(UInt160 owner, ByteString tokenId, BigInteger increment)
        {
            var key = owner + tokenId;
            StorageMap balanceMap = new(Storage.CurrentContext, prefixBalance);
            BigInteger balance = (BigInteger)balanceMap[key];
            balance += increment;
            
            if (balance < 0)
            {
                throw new InvalidOperationException("Balance underflow execption");
            }
            
            if (balance.IsZero)
            {
                balanceMap.Delete(owner);
            } else {
                balanceMap.Put(key, balance);
            }

            StorageMap accountMap = new(Storage.CurrentContext, prefixAccountToken);

            if (increment > 0) 
            {
                accountMap.Put(key, 0);
            } else {
                accountMap.Delete(key);
            }
        }
    }
}